/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checks;

import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.tools.ImportsFormatter;
import com.liferay.source.formatter.BNDImportsFormatter;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class BNDImportsCheck extends BaseFileCheck {

	@Override
	protected String doProcess(
			String fileName, String absolutePath, String content)
		throws Exception {

		_checkWildcardImports(
			fileName, absolutePath, content, _conditionalPackagePattern);
		_checkWildcardImports(
			fileName, absolutePath, content, _exportContentsPattern);
		_checkWildcardImports(fileName, absolutePath, content, _exportsPattern);

		ImportsFormatter importsFormatter = new BNDImportsFormatter();

		content = importsFormatter.format(content, _conditionalPackagePattern);
		content = importsFormatter.format(content, _exportContentsPattern);
		content = importsFormatter.format(content, _exportsPattern);
		content = importsFormatter.format(content, _importsPattern);
		content = importsFormatter.format(content, _privatePackagesPattern);

		if (!absolutePath.contains("-test/")) {
			content = _removeInternalPrivatePackages(content);
		}

		return content;
	}

	private void _checkWildcardImports(
		String fileName, String absolutePath, String content, Pattern pattern) {

		if (absolutePath.contains("/portal-kernel/") ||
			absolutePath.contains("/support-tomcat/") ||
			absolutePath.contains("/third-party/") ||
			absolutePath.contains("/util-bridges/") ||
			absolutePath.contains("/util-java/") ||
			absolutePath.contains("/util-taglib/") ||
			fileName.endsWith("/system.packages.extra.bnd")) {

			return;
		}

		Matcher matcher = pattern.matcher(content);

		if (!matcher.find()) {
			return;
		}

		String imports = matcher.group(3);

		matcher = _wilcardImportPattern.matcher(imports);

		while (matcher.find()) {
			String wildcardImport = matcher.group(1);

			if (wildcardImport.matches("^!?com\\.liferay\\..+")) {
				addMessage(
					fileName,
					"Do not use wildcard in Export-Package '" + wildcardImport +
						"'");
			}
		}
	}

	private String _removeInternalPrivatePackages(String content) {
		Matcher matcher = _privatePackagesPattern.matcher(content);

		if (!matcher.find()) {
			return content;
		}

		String match = matcher.group();

		matcher = _internalPrivatePackagePattern.matcher(match);

		if (!matcher.find()) {
			return content;
		}

		String replacement = StringUtil.removeSubstring(
			match, matcher.group(2));

		return StringUtil.replace(content, match, replacement);
	}

	private final Pattern _conditionalPackagePattern = Pattern.compile(
		"\n-conditionalpackage:(\\\\\n| )((.*?)(\n[^\t]|\\Z))",
		Pattern.DOTALL | Pattern.MULTILINE);
	private final Pattern _exportContentsPattern = Pattern.compile(
		"\n-exportcontents:(\\\\\n| )((.*?)(\n[^\t]|\\Z))",
		Pattern.DOTALL | Pattern.MULTILINE);
	private final Pattern _exportsPattern = Pattern.compile(
		"\nExport-Package:(\\\\\n| )((.*?)(\n[^\t]|\\Z))",
		Pattern.DOTALL | Pattern.MULTILINE);
	private final Pattern _importsPattern = Pattern.compile(
		"\nImport-Package:(\\\\\n| )((.*?)(\n[^\t]|\\Z))",
		Pattern.DOTALL | Pattern.MULTILINE);
	private final Pattern _internalPrivatePackagePattern = Pattern.compile(
		"(,\\\\\n\t|: )(.*\\.internal.*)(\n|\\Z)");
	private final Pattern _privatePackagesPattern = Pattern.compile(
		"\nPrivate-Package:(\\\\\n| )((.*?)(\n[^\t]|\\Z))",
		Pattern.DOTALL | Pattern.MULTILINE);
	private final Pattern _wilcardImportPattern = Pattern.compile(
		"(\\S+\\*)(,\\\\\n|\n|\\Z)");

}