/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.check;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.source.formatter.util.LegacyPropertiesUtil;
import com.liferay.source.formatter.util.LegacyProperty;
import com.liferay.source.formatter.util.LegacyPropertyAction;
import com.liferay.source.formatter.util.LegacyPropertyType;

import java.io.StringReader;

import java.util.List;
import java.util.Properties;

/**
 * @author Hugo Huijser
 */
public class PropertiesVerifyPropertiesCheck extends BaseFileCheck {

	@Override
	public boolean isLiferaySourceCheck() {
		return true;
	}

	@Override
	protected String doProcess(
			String fileName, String absolutePath, String content)
		throws Exception {

		if (absolutePath.contains("/modules/")) {
			return content;
		}

		LegacyPropertyType legacyPropertyType = null;

		if (absolutePath.matches(".*/portal([-\\w]+)?\\.properties")) {
			legacyPropertyType = LegacyPropertyType.PORTAL;
		}
		else if (absolutePath.matches(".*/system([-\\w]+)?\\.properties")) {
			legacyPropertyType = LegacyPropertyType.SYSTEM;
		}
		else {
			return content;
		}

		Properties properties = new Properties();

		properties.load(new StringReader(content));

		List<LegacyProperty> legacyProperties = _getLegacyProperties(
			absolutePath);

		for (LegacyProperty legacyProperty : legacyProperties) {
			if (!properties.containsKey(
					legacyProperty.getLegacyPropertyName()) ||
				!legacyPropertyType.equals(
					legacyProperty.getLegacyPropertyType())) {

				continue;
			}

			StringBundler sb = new StringBundler(10);

			sb.append(legacyPropertyType.getValue());
			sb.append(" property '");
			sb.append(legacyProperty.getLegacyPropertyName());

			LegacyPropertyAction legacyActionType =
				legacyProperty.getLegacyPropertyAction();

			if (legacyActionType.equals(LegacyPropertyAction.MIGRATED)) {
				sb.append("' was migrated to ");

				if (legacyPropertyType.equals(LegacyPropertyType.PORTAL)) {
					sb.append("'system.properties'");
				}
				else {
					sb.append("'portal.properties'");
				}
			}
			else if (legacyActionType.equals(
						LegacyPropertyAction.MODULARIZED)) {

				sb.append("' was modularized");
			}
			else if (legacyActionType.equals(LegacyPropertyAction.OBSOLETE)) {
				sb.append("' is obsolete");
			}
			else {
				sb.append("' was renamed");
			}

			sb.append(". See '");
			sb.append(
				StringUtil.removeSubstring(
					_VERIFY_PROPERTIES_FILE_NAME, ".java"));
			sb.append("#");
			sb.append(legacyProperty.getVariableName());
			sb.append("'.");

			addMessage(fileName, sb.toString());
		}

		return content;
	}

	private synchronized List<LegacyProperty> _getLegacyProperties(
			String absolutePath)
		throws Exception {

		if (_legacyProperties != null) {
			return _legacyProperties;
		}

		_legacyProperties = LegacyPropertiesUtil.getLegacyProperties(
			_VERIFY_PROPERTIES_FILE_NAME,
			getPortalContent(_VERIFY_PROPERTIES_FILE_NAME, absolutePath));

		return _legacyProperties;
	}

	private static final String _VERIFY_PROPERTIES_FILE_NAME =
		"portal-impl/src/com/liferay/portal/verify/VerifyProperties.java";

	private List<LegacyProperty> _legacyProperties;

}