/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.check;

import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.tools.ToolsUtil;
import com.liferay.source.formatter.check.util.JavaSourceUtil;
import com.liferay.source.formatter.parser.JavaClass;
import com.liferay.source.formatter.parser.JavaTerm;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class JavaStringBundlerConcatCheck extends BaseJavaTermCheck {

	@Override
	public boolean isLiferaySourceCheck() {
		return true;
	}

	@Override
	protected String doProcess(
		String fileName, String absolutePath, JavaTerm javaTerm,
		String fileContent) {

		JavaClass javaClass = (JavaClass)javaTerm;

		List<String> imports = javaClass.getImportNames();

		boolean hasPetraStringStringBundler = imports.contains(
			"com.liferay.petra.string.StringBundler");

		String content = javaTerm.getContent();

		Matcher matcher1 = _stringBundlerConcatPattern.matcher(content);

		while (matcher1.find()) {
			if (ToolsUtil.isInsideQuotes(content, matcher1.start() + 1)) {
				continue;
			}

			String stringBundlerConcatMethodCall = JavaSourceUtil.getMethodCall(
				content, matcher1.start());

			List<String> parameterList = JavaSourceUtil.getParameterList(
				stringBundlerConcatMethodCall);

			if (parameterList.size() < 3) {
				addMessage(
					fileName,
					"Do not use 'StringBundler.concat' when concatenating " +
						"less than 3 elements",
					javaTerm.getLineNumber(matcher1.start()));
			}

			if (!hasPetraStringStringBundler) {
				continue;
			}

			Matcher matcher2 = _stringValueOfPattern.matcher(
				stringBundlerConcatMethodCall);

			while (matcher2.find()) {
				int level = getLevel(
					stringBundlerConcatMethodCall.substring(
						0, matcher2.start()));

				if (level != 1) {
					continue;
				}

				String stringValueOfMethodCall = JavaSourceUtil.getMethodCall(
					stringBundlerConcatMethodCall, matcher2.start());

				parameterList = JavaSourceUtil.getParameterList(
					stringValueOfMethodCall);

				if (parameterList.size() != 1) {
					continue;
				}

				String newStringBundlerConcatMethodCall =
					StringUtil.replaceFirst(
						stringBundlerConcatMethodCall, stringValueOfMethodCall,
						parameterList.get(0));

				return StringUtil.replace(
					content, stringBundlerConcatMethodCall,
					newStringBundlerConcatMethodCall);
			}
		}

		return javaTerm.getContent();
	}

	@Override
	protected String[] getCheckableJavaTermNames() {
		return new String[] {JAVA_CLASS};
	}

	private static final Pattern _stringBundlerConcatPattern = Pattern.compile(
		"StringBundler\\.concat\\(");
	private static final Pattern _stringValueOfPattern = Pattern.compile(
		"String\\.valueOf\\(");

}