/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.check;

import com.liferay.petra.string.StringPool;
import com.liferay.source.formatter.check.util.JavaSourceUtil;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class JavaModuleServiceReferenceCheck extends BaseFileCheck {

	@Override
	public boolean isModuleSourceCheck() {
		return true;
	}

	@Override
	protected String doProcess(
		String fileName, String absolutePath, String content) {

		String packageName = JavaSourceUtil.getPackageName(content);

		int pos = packageName.indexOf(".service.");

		if (pos == -1) {
			return content;
		}

		String servicePackageName = packageName.substring(0, pos + 8);

		_checkServiceReferences(
			fileName, content, packageName, servicePackageName);

		return content;
	}

	private void _checkServiceReferences(
		String fileName, String content, String packageName,
		String servicePackageName) {

		Matcher matcher = _serviceReferencePattern.matcher(content);

		while (matcher.find()) {
			String className = _getFullClassName(
				content, matcher.group(1), packageName);

			if (className.startsWith(servicePackageName)) {
				addMessage(
					fileName, "Use @BeanReference instead of @ServiceReference",
					getLineNumber(content, matcher.start()));
			}
		}
	}

	private String _getFullClassName(
		String content, String className, String packageName) {

		if (className.contains(StringPool.PERIOD)) {
			return className;
		}

		Pattern pattern = Pattern.compile("import (.*" + className + ");");

		Matcher matcher = pattern.matcher(content);

		if (matcher.find()) {
			return matcher.group(1);
		}

		return packageName + StringPool.PERIOD + className;
	}

	private static final Pattern _serviceReferencePattern = Pattern.compile(
		"@ServiceReference\\(\\s*type = ([\\w.]+)\\.class\\)\n");

}