/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.poshi.core.script;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Calum Ragan
 */
public class PoshiScriptParserUtil {

	public static List<String> getMethodParameterValues(String content) {
		try {
			return getMethodParameterValues(content, null);
		}
		catch (PoshiScriptParserException poshiScriptParserException) {
			poshiScriptParserException.printStackTrace();

			return new ArrayList<>();
		}
	}

	public static List<String> getMethodParameterValues(
			String content, Pattern pattern)
		throws PoshiScriptParserException {

		List<String> methodParameterValues = new ArrayList<>();

		if (content.length() == 0) {
			return methodParameterValues;
		}

		StringBuilder sb = new StringBuilder();

		String methodParameterValue = sb.toString();

		for (char c : content.toCharArray()) {
			if ((c == ',') && isBalancedPoshiScript(methodParameterValue)) {
				if (pattern != null) {
					Matcher matcher = pattern.matcher(methodParameterValue);

					if (!matcher.matches()) {
						sb.append(c);

						continue;
					}
				}

				methodParameterValues.add(methodParameterValue);

				sb.setLength(0);

				continue;
			}

			sb.append(c);

			methodParameterValue = sb.toString();
		}

		if (pattern != null) {
			Matcher matcher = pattern.matcher(methodParameterValue);

			if (!matcher.matches()) {
				throw new PoshiScriptParserException(
					"Invalid Poshi Script parameter syntax");
			}
		}

		methodParameterValues.add(methodParameterValue);

		return methodParameterValues;
	}

	public static boolean isBalancedPoshiScript(String poshiScript) {
		try {
			return isBalancedPoshiScript(poshiScript, false);
		}
		catch (Exception exception) {
			return false;
		}
	}

	public static boolean isBalancedPoshiScript(
			String poshiScript, boolean throwException)
		throws UnbalancedCodeException {

		poshiScript = _fixPoshiScript(poshiScript);

		Stack<Integer> stack = new Stack<>();

		for (int i = 0; i < poshiScript.length(); i++) {
			char c = poshiScript.charAt(i);

			if (!stack.isEmpty()) {
				int topIndex = stack.peek();

				Character topCodeBoundary = poshiScript.charAt(topIndex);

				if ((c == _codeBoundariesMap.get(topCodeBoundary)) && (i > 0) &&
					(poshiScript.charAt(i - 1) != '\\')) {

					stack.pop();

					continue;
				}

				if ((topCodeBoundary == '\"') || (topCodeBoundary == '\'')) {
					continue;
				}
			}

			if (_codeBoundariesMap.containsKey(c)) {
				stack.push(i);

				continue;
			}

			if (_codeBoundariesMap.containsValue(c)) {
				if (throwException) {
					throw new UnbalancedCodeException(
						"Unexpected closing boundary", i, poshiScript);
				}

				return false;
			}
		}

		boolean balanced = stack.isEmpty();

		if (!balanced && throwException) {
			throw new UnbalancedCodeException(
				"Unmatched opening boundary", stack.peek(), poshiScript);
		}

		return balanced;
	}

	private static String _fixPoshiScript(String poshiScript) {
		if (poshiScript.contains("/*") && poshiScript.contains("*/")) {
			poshiScript = poshiScript.replaceAll("(?s)/\\*.*?\\*/", "/\\*\\*/");
		}

		if (poshiScript.contains("'''")) {
			poshiScript = poshiScript.replaceAll(
				"(?s)\'\'\'.*?\'\'\'", "\'\'\'\'\'\'");
		}

		if (poshiScript.contains("//")) {
			poshiScript = poshiScript.replaceAll("(?m)\n[\\s]*//.*?$", "//\n");
		}

		return poshiScript.trim();
	}

	private static final Map<Character, Character> _codeBoundariesMap =
		new HashMap<Character, Character>() {
			{
				put('(', ')');
				put('[', ']');
				put('\"', '\"');
				put('\'', '\'');
				put('{', '}');
			}
		};

}