/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checkstyle.checks;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

import java.util.List;

/**
 * @author Hugo Huijser
 */
public class AssignAsUsedCheck extends BaseAsUsedCheck {

	@Override
	public int[] getDefaultTokens() {
		return new int[] {TokenTypes.CTOR_DEF, TokenTypes.METHOD_DEF};
	}

	@Override
	protected void doVisitToken(DetailAST detailAST) {
		List<DetailAST> assignDetailASTList = getAllChildTokens(
			detailAST, true, TokenTypes.ASSIGN);

		for (DetailAST assignDetailAST : assignDetailASTList) {
			DetailAST parentDetailAST = assignDetailAST.getParent();

			if (parentDetailAST.getType() != TokenTypes.EXPR) {
				continue;
			}

			parentDetailAST = parentDetailAST.getParent();

			if (parentDetailAST.getType() == TokenTypes.SLIST) {
				_checkAssign(
					detailAST, assignDetailAST,
					getEndLineNumber(parentDetailAST));
			}
		}
	}

	private void _checkAssign(
		DetailAST detailAST, DetailAST assignDetailAST, int endRange) {

		if (hasParentWithTokenType(
				assignDetailAST, TokenTypes.FOR_EACH_CLAUSE,
				TokenTypes.FOR_INIT)) {

			return;
		}

		DetailAST nameDetailAST = assignDetailAST.findFirstToken(
			TokenTypes.IDENT);

		if (nameDetailAST == null) {
			return;
		}

		String variableName = nameDetailAST.getText();

		DetailAST typeDetailAST = getVariableTypeDetailAST(
			assignDetailAST, variableName, false);

		if (typeDetailAST == null) {
			return;
		}

		DetailAST parentDetailAST = typeDetailAST.getParent();

		if (parentDetailAST.getType() != TokenTypes.VARIABLE_DEF) {
			return;
		}

		List<DetailAST> dependentIdentDetailASTList =
			getDependentIdentDetailASTList(
				parentDetailAST, parentDetailAST.getLineNo());

		if (dependentIdentDetailASTList.isEmpty()) {
			return;
		}

		int endLineNumber = getEndLineNumber(assignDetailAST);

		for (DetailAST dependentIdentDetailAST : dependentIdentDetailASTList) {
			int lineNumber = dependentIdentDetailAST.getLineNo();

			if (lineNumber <= endLineNumber) {
				continue;
			}

			if (lineNumber > endRange) {
				return;
			}

			if (!hasParentWithTokenType(
					assignDetailAST, TokenTypes.LITERAL_FOR,
					TokenTypes.LITERAL_WHILE)) {

				int actionLineNumber = getActionLineNumber(assignDetailAST);

				if (actionLineNumber != assignDetailAST.getLineNo()) {
					checkMoveAfterBranchingStatement(
						detailAST, assignDetailAST, variableName,
						dependentIdentDetailAST, actionLineNumber);
					checkMoveInsideIfStatement(
						assignDetailAST, nameDetailAST, variableName,
						dependentIdentDetailAST,
						dependentIdentDetailASTList.get(
							dependentIdentDetailASTList.size() - 1),
						actionLineNumber);
				}
			}

			checkInline(
				assignDetailAST, variableName, dependentIdentDetailAST,
				dependentIdentDetailASTList);

			return;
		}
	}

}