/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checks;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.source.formatter.checks.util.SourceUtil;
import com.liferay.source.formatter.util.FileUtil;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;

import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Alan Huang
 */
public class PoshiDependenciesFileLocationCheck extends BaseFileCheck {

	@Override
	protected String doProcess(
			String fileName, String absolutePath, String content)
		throws IOException {

		if (!fileName.endsWith(".testcase")) {
			return content;
		}

		_getTestCaseFileNames();

		_checkDependenciesFileReferences(absolutePath, fileName);

		_checkGlobalDependenciesFileReferences(absolutePath, fileName);

		return content;
	}

	private synchronized void _checkDependenciesFileReferences(
			String absolutePath, String fileName)
		throws IOException {

		_getTestCaseDependenciesFileLocations();

		if (!_dependenciesFileLocationsMapIsReady) {
			for (String testCaseFileName : _testCaseFileNames) {
				File testCaseFile = new File(testCaseFileName);

				String testCaseFileContent = FileUtil.read(testCaseFile);

				for (Map.Entry<String, Set<String>> entry :
						_dependenciesFileLocationsMap.entrySet()) {

					String dependenciesFileLocation = entry.getKey();

					String dependenciesFileName =
						dependenciesFileLocation.replaceFirst(".*/(.+)", "$1");

					String s = Pattern.quote(dependenciesFileName);

					Pattern dependenciesFileNamePattern = Pattern.compile(
						"[\",]" + s + "[\",]");

					Matcher matcher = dependenciesFileNamePattern.matcher(
						testCaseFileContent);

					while (matcher.find()) {
						Set<String> referencesFiles = entry.getValue();

						referencesFiles.add(testCaseFileName);

						_dependenciesFileLocationsMap.put(
							dependenciesFileLocation, referencesFiles);
					}
				}
			}
		}

		for (Map.Entry<String, Set<String>> entry :
				_dependenciesFileLocationsMap.entrySet()) {

			Set<String> referencesFiles = entry.getValue();

			Set<String> removedDuplicatedFilePaths = new HashSet<>();

			for (String referencesFile : referencesFiles) {
				String referencesFilePath = referencesFile.substring(
					0, referencesFile.lastIndexOf("/"));

				removedDuplicatedFilePaths.add(referencesFilePath);
			}

			if (removedDuplicatedFilePaths.size() <= 1) {
				continue;
			}

			if (referencesFiles.size() > 1) {
				for (String referencesFile : referencesFiles) {
					if (referencesFile.equals(absolutePath)) {
						addMessage(
							fileName,
							StringBundler.concat(
								"Test dependencies file '", entry.getKey(),
								"' is referenced by multiple modules, move it ",
								"to global dependencies directory"));

						break;
					}
				}
			}
		}

		_dependenciesFileLocationsMapIsReady = true;
	}

	private synchronized void _checkGlobalDependenciesFileReferences(
			String absolutePath, String fileName)
		throws IOException {

		_getTestCaseGlobalDependenciesFileLocations();

		if (!_dependenciesGlobalFileLocationsMapIsReady) {
			for (String testCaseFileName : _testCaseFileNames) {
				File testCaseFile = new File(testCaseFileName);

				String testCaseFileContent = FileUtil.read(testCaseFile);

				for (Map.Entry<String, Set<String>> entry :
						_dependenciesGlobalFileLocationsMap.entrySet()) {

					String dependenciesFileLocation = entry.getKey();

					String dependenciesFileName =
						dependenciesFileLocation.replaceFirst(".*/(.+)", "$1");

					String s = Pattern.quote(dependenciesFileName);

					Pattern dependenciesFileNamePattern = Pattern.compile(
						"[\",]" + s + "[\",]");

					Matcher matcher = dependenciesFileNamePattern.matcher(
						testCaseFileContent);

					while (matcher.find()) {
						Set<String> referencesFiles = entry.getValue();

						referencesFiles.add(testCaseFileName);

						_dependenciesGlobalFileLocationsMap.put(
							dependenciesFileLocation, referencesFiles);
					}
				}
			}
		}

		for (Map.Entry<String, Set<String>> entry :
				_dependenciesGlobalFileLocationsMap.entrySet()) {

			Set<String> referencesFiles = entry.getValue();

			if (referencesFiles.size() == 1) {
				for (String referencesFile : referencesFiles) {
					if (referencesFile.equals(absolutePath)) {
						addMessage(
							fileName,
							StringBundler.concat(
								"Test dependencies file '", entry.getKey(),
								"' is only referenced by one module, move it ",
								"to module dependencies directory"));

						break;
					}
				}
			}
		}

		_dependenciesGlobalFileLocationsMapIsReady = true;
	}

	private synchronized void _getTestCaseDependenciesFileLocations()
		throws IOException {

		if (!_dependenciesFileLocationsMap.isEmpty()) {
			return;
		}

		for (String dependenciesFileLocation : _TEST_FILE_LOCATIONS) {
			File directory = new File(getPortalDir(), dependenciesFileLocation);

			Path dirPath = directory.toPath();

			Files.walkFileTree(
				dirPath, EnumSet.noneOf(FileVisitOption.class), 25,
				new SimpleFileVisitor<Path>() {

					@Override
					public FileVisitResult preVisitDirectory(
							Path dirPath,
							BasicFileAttributes basicFileAttributes)
						throws IOException {

						if (ArrayUtil.contains(
								_SKIP_DIR_NAMES,
								String.valueOf(dirPath.getFileName()))) {

							return FileVisitResult.SKIP_SUBTREE;
						}

						String absolutePath = SourceUtil.getAbsolutePath(
							dirPath);

						if (absolutePath.contains("/test/") ||
							absolutePath.contains("/tests/")) {

							if (absolutePath.endsWith("/dependencies")) {
								File dirFile = dirPath.toFile();

								File[] dependenciesFiles = dirFile.listFiles(
									new FileFilter() {

										@Override
										public boolean accept(File file) {
											if (!file.isFile()) {
												return false;
											}

											return true;
										}

									});

								for (File dependenciesFile :
										dependenciesFiles) {

									_dependenciesFileLocationsMap.put(
										SourceUtil.getAbsolutePath(
											dependenciesFile.getPath()),
										new TreeSet<>());
								}
							}

							if (absolutePath.matches(
									".+/dependencies/.+\\..+")) {

								_dependenciesFileLocationsMap.put(
									SourceUtil.getAbsolutePath(absolutePath),
									new TreeSet<>());

								return FileVisitResult.SKIP_SUBTREE;
							}
						}

						return FileVisitResult.CONTINUE;
					}

				});
		}
	}

	private synchronized void _getTestCaseFileNames() throws IOException {
		if (!_testCaseFileNames.isEmpty()) {
			return;
		}

		for (String testCaseFileLocation : _TEST_FILE_LOCATIONS) {
			File directory = new File(getPortalDir(), testCaseFileLocation);

			Path dirPath = directory.toPath();

			Files.walkFileTree(
				dirPath, EnumSet.noneOf(FileVisitOption.class), 25,
				new SimpleFileVisitor<Path>() {

					@Override
					public FileVisitResult preVisitDirectory(
							Path dirPath,
							BasicFileAttributes basicFileAttributes)
						throws IOException {

						if (ArrayUtil.contains(
								_SKIP_DIR_NAMES,
								String.valueOf(dirPath.getFileName()))) {

							return FileVisitResult.SKIP_SUBTREE;
						}

						String absolutePath = SourceUtil.getAbsolutePath(
							dirPath);

						if (!absolutePath.contains("portal-web") &&
							!absolutePath.matches(
								".+/modules/.+-test/src/testFunctional(/.*)" +
									"?")) {

							return FileVisitResult.CONTINUE;
						}

						File dirFile = dirPath.toFile();

						File[] testcaseFiles = dirFile.listFiles(
							new FileFilter() {

								@Override
								public boolean accept(File file) {
									if (!file.isFile()) {
										return false;
									}

									String fileName = file.getName();

									if (fileName.endsWith(".testcase")) {
										return true;
									}

									return false;
								}

							});

						for (File testcaseFile : testcaseFiles) {
							_testCaseFileNames.add(
								SourceUtil.getAbsolutePath(
									testcaseFile.getPath()));
						}

						return FileVisitResult.CONTINUE;
					}

				});
		}
	}

	private synchronized void _getTestCaseGlobalDependenciesFileLocations()
		throws IOException {

		if (!_dependenciesGlobalFileLocationsMap.isEmpty()) {
			return;
		}

		File directory = new File(
			getPortalDir(), _GLOBAL_DEPENDENCIES_DIRECTORY);

		Path dirPath = directory.toPath();

		Files.walkFileTree(
			dirPath, EnumSet.noneOf(FileVisitOption.class), 25,
			new SimpleFileVisitor<Path>() {

				@Override
				public FileVisitResult preVisitDirectory(
						Path dirPath, BasicFileAttributes basicFileAttributes)
					throws IOException {

					if (ArrayUtil.contains(
							_SKIP_DIR_NAMES,
							String.valueOf(dirPath.getFileName()))) {

						return FileVisitResult.SKIP_SUBTREE;
					}

					String absolutePath = SourceUtil.getAbsolutePath(dirPath);

					if (absolutePath.matches(".+/dependencies/.+\\..+")) {
						_dependenciesGlobalFileLocationsMap.put(
							SourceUtil.getAbsolutePath(absolutePath),
							new TreeSet<>());

						return FileVisitResult.SKIP_SUBTREE;
					}

					File dirFile = dirPath.toFile();

					File[] dependenciesFiles = dirFile.listFiles(
						new FileFilter() {

							@Override
							public boolean accept(File file) {
								if (!file.isFile()) {
									return false;
								}

								return true;
							}

						});

					for (File dependenciesFile : dependenciesFiles) {
						_dependenciesGlobalFileLocationsMap.put(
							SourceUtil.getAbsolutePath(
								dependenciesFile.getPath()),
							new TreeSet<>());
					}

					return FileVisitResult.CONTINUE;
				}

			});
	}

	private static final String _GLOBAL_DEPENDENCIES_DIRECTORY =
		"portal-web/test/functional/com/liferay/portalweb/dependencies";

	private static final String[] _SKIP_DIR_NAMES = {
		".git", ".gradle", ".idea", ".m2", ".releng", ".settings", "bin",
		"build", "classes", "node_modules", "node_modules_cache", "poshi",
		"private", "source-formatter"
	};

	private static final String[] _TEST_FILE_LOCATIONS = {
		"modules", "portal-web/test/functional/com/liferay/portalweb/tests"
	};

	private static final Map<String, Set<String>>
		_dependenciesFileLocationsMap = new HashMap<>();
	private static boolean _dependenciesFileLocationsMapIsReady;
	private static final Map<String, Set<String>>
		_dependenciesGlobalFileLocationsMap = new HashMap<>();
	private static boolean _dependenciesGlobalFileLocationsMapIsReady;
	private static final List<String> _testCaseFileNames = new ArrayList<>();

}