/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checks;

import com.liferay.petra.string.StringBundler;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.source.formatter.parser.JavaTerm;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class JavaComponentActivateCheck extends BaseJavaTermCheck {

	@Override
	public boolean isLiferaySourceCheck() {
		return true;
	}

	@Override
	protected String doProcess(
		String fileName, String absolutePath, JavaTerm javaTerm,
		String fileContent) {

		List<String> importNames = getImportNames(javaTerm);

		if (importNames.isEmpty()) {
			return javaTerm.getContent();
		}

		return _formatModifiers(
			fileName, javaTerm, importNames, "Activate", "Deactivate");
	}

	@Override
	protected String[] getCheckableJavaTermNames() {
		return new String[] {JAVA_METHOD};
	}

	private String _formatModifiers(
		String fileName, JavaTerm javaTerm, List<String> importNames,
		String... annotationNames) {

		String content = javaTerm.getContent();

		for (String annotationName : annotationNames) {
			if (!javaTerm.hasAnnotation(annotationName) ||
				!importNames.contains(
					"org.osgi.service.component.annotations." +
						annotationName)) {

				continue;
			}

			String methodName = javaTerm.getName();

			if (!javaTerm.hasAnnotation("Override")) {
				String expectedMethodName = StringUtil.toLowerCase(
					annotationName);

				if (!methodName.equals(expectedMethodName)) {
					addMessage(
						fileName,
						StringBundler.concat(
							"Method with annotation '", annotationName,
							"' should have name '", expectedMethodName, "'"));
				}
			}

			if (javaTerm.isProtected()) {
				continue;
			}

			Pattern pattern = Pattern.compile(
				StringBundler.concat(
					"(\\s)", javaTerm.getAccessModifier(), "( (.* )?void\\s*",
					javaTerm.getName(), ")"));

			Matcher matcher = pattern.matcher(content);

			if (matcher.find()) {
				return StringUtil.replaceFirst(
					content, javaTerm.getAccessModifier(),
					JavaTerm.ACCESS_MODIFIER_PROTECTED, matcher.start());
			}

			addMessage(
				fileName,
				"Method '" + javaTerm.getName() + "' should be protected",
				javaTerm.getLineNumber(matcher.start()));
		}

		return content;
	}

}