/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.tools.bundle.support.internal.util;

import com.liferay.portal.tools.bundle.support.util.StreamLogger;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.net.InetAddress;
import java.net.URI;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;

import java.util.Collections;
import java.util.Date;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.DateUtils;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.RedirectLocations;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

/**
 * @author David Truong
 * @author Andrea Di Giorgi
 */
public class HttpUtil {

	public static String createToken(URI uri, String userName, String password)
		throws IOException {

		try (CloseableHttpClient closeableHttpClient = _getHttpClient(
				uri, userName, password, -1)) {

			HttpPost httpPost = new HttpPost(uri);

			InetAddress inetAddress = InetAddress.getLocalHost();

			NameValuePair deviceNameValuePair = new BasicNameValuePair(
				"device",
				"portal-tools-bundle-support-" + inetAddress.getHostName());

			httpPost.setEntity(
				new UrlEncodedFormEntity(
					Collections.singleton(deviceNameValuePair)));

			try (CloseableHttpResponse closeableHttpResponse =
					closeableHttpClient.execute(httpPost)) {

				_checkResponseStatus(closeableHttpResponse);

				HttpEntity httpEntity = closeableHttpResponse.getEntity();

				String json = EntityUtils.toString(
					httpEntity, StandardCharsets.UTF_8);

				int start = json.indexOf("token\":") + 7;

				start = json.indexOf('"', start) + 1;

				int end = json.indexOf('"', start);

				return json.substring(start, end);
			}
		}
	}

	public static Path downloadFile(
			URI uri, String token, Path cacheDirPath, StreamLogger streamLogger)
		throws Exception {

		return downloadFile(uri, token, cacheDirPath, streamLogger, -1);
	}

	public static Path downloadFile(
			URI uri, String token, Path cacheDirPath, StreamLogger streamLogger,
			int connectionTimeout)
		throws Exception {

		Path path;

		try (CloseableHttpClient closeableHttpClient = _getHttpClient(
				uri, token, connectionTimeout)) {

			path = _downloadFile(
				closeableHttpClient, uri, cacheDirPath, streamLogger);
		}

		return path;
	}

	public static Path downloadFile(
			URI uri, String userName, String password, Path cacheDirPath,
			StreamLogger streamLogger)
		throws Exception {

		return downloadFile(
			uri, userName, password, cacheDirPath, streamLogger, -1);
	}

	public static Path downloadFile(
			URI uri, String userName, String password, Path cacheDirPath,
			StreamLogger streamLogger, int connectionTimeout)
		throws Exception {

		Path path;

		try (CloseableHttpClient closeableHttpClient = _getHttpClient(
				uri, userName, password, connectionTimeout)) {

			path = _downloadFile(
				closeableHttpClient, uri, cacheDirPath, streamLogger);
		}

		return path;
	}

	private static void _checkResponseStatus(HttpResponse httpResponse)
		throws IOException {

		StatusLine statusLine = httpResponse.getStatusLine();

		if (statusLine.getStatusCode() != HttpStatus.SC_OK) {
			throw new IOException(statusLine.getReasonPhrase());
		}
	}

	private static Path _downloadFile(
			CloseableHttpClient closeableHttpClient, URI uri, Path cacheDirPath,
			StreamLogger streamLogger)
		throws Exception {

		HttpHead httpHead = new HttpHead(uri);

		HttpContext httpContext = new BasicHttpContext();

		String fileName = null;

		Date lastModifiedDate;

		try (CloseableHttpResponse closeableHttpResponse =
				closeableHttpClient.execute(httpHead, httpContext)) {

			_checkResponseStatus(closeableHttpResponse);

			Header dispositionHeader = closeableHttpResponse.getFirstHeader(
				"Content-Disposition");

			if (dispositionHeader != null) {
				String dispositionValue = dispositionHeader.getValue();

				int index = dispositionValue.indexOf("filename=");

				if (index > 0) {
					fileName = dispositionValue.substring(
						index + "filename=".length());
				}
			}
			else {
				RedirectLocations redirectLocations =
					(RedirectLocations)httpContext.getAttribute(
						HttpClientContext.REDIRECT_LOCATIONS);

				if (redirectLocations != null) {
					uri = redirectLocations.get(redirectLocations.size() - 1);
				}
			}

			Header lastModifiedHeader = closeableHttpResponse.getFirstHeader(
				HttpHeaders.LAST_MODIFIED);

			if (lastModifiedHeader != null) {
				lastModifiedDate = DateUtils.parseDate(
					lastModifiedHeader.getValue());
			}
			else {
				lastModifiedDate = new Date();
			}
		}

		if (fileName == null) {
			String uriPath = uri.getPath();

			fileName = uriPath.substring(uriPath.lastIndexOf('/') + 1);
		}

		if (cacheDirPath == null) {
			cacheDirPath = Files.createTempDirectory(null);
		}

		Path path = cacheDirPath.resolve(fileName);

		if (Files.exists(path)) {
			FileTime fileTime = Files.getLastModifiedTime(path);

			if (fileTime.toMillis() == lastModifiedDate.getTime()) {
				return path;
			}

			Files.delete(path);
		}

		Files.createDirectories(cacheDirPath);

		HttpGet httpGet = new HttpGet(uri);

		try (CloseableHttpResponse closeableHttpResponse =
				closeableHttpClient.execute(httpGet)) {

			_checkResponseStatus(closeableHttpResponse);

			HttpEntity httpEntity = closeableHttpResponse.getEntity();

			long length = httpEntity.getContentLength();

			streamLogger.onStarted();

			try (InputStream inputStream = httpEntity.getContent();
				OutputStream outputStream = Files.newOutputStream(path)) {

				byte[] buffer = new byte[10 * 1024];
				int completed = 0;
				int read = -1;

				while ((read = inputStream.read(buffer)) >= 0) {
					outputStream.write(buffer, 0, read);

					completed += read;

					streamLogger.onProgress(completed, length);
				}
			}
			finally {
				streamLogger.onCompleted();
			}
		}

		Files.setLastModifiedTime(
			path, FileTime.fromMillis(lastModifiedDate.getTime()));

		return path;
	}

	private static CloseableHttpClient _getHttpClient(
		URI uri, String token, int connectionTimeout) {

		HttpClientBuilder httpClientBuilder = _getHttpClientBuilder(
			uri, null, null, connectionTimeout);

		Header header = new BasicHeader(
			HttpHeaders.AUTHORIZATION, "Bearer " + token);

		httpClientBuilder.setDefaultHeaders(Collections.singleton(header));

		return httpClientBuilder.build();
	}

	private static CloseableHttpClient _getHttpClient(
		URI uri, String userName, String password, int connectionTimeout) {

		HttpClientBuilder httpClientBuilder = _getHttpClientBuilder(
			uri, userName, password, connectionTimeout);

		return httpClientBuilder.build();
	}

	private static HttpClientBuilder _getHttpClientBuilder(
		URI uri, String userName, String password, int connectionTimeout) {

		HttpClientBuilder httpClientBuilder = HttpClients.custom();

		CredentialsProvider credentialsProvider =
			new BasicCredentialsProvider();

		httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);

		RequestConfig.Builder requestConfigBuilder = RequestConfig.custom();

		requestConfigBuilder.setConnectTimeout(connectionTimeout);
		requestConfigBuilder.setCookieSpec(CookieSpecs.STANDARD);
		requestConfigBuilder.setRedirectsEnabled(true);

		httpClientBuilder.setDefaultRequestConfig(requestConfigBuilder.build());

		if ((userName != null) && (password != null)) {
			credentialsProvider.setCredentials(
				new AuthScope(uri.getHost(), uri.getPort()),
				new UsernamePasswordCredentials(userName, password));
		}

		String scheme = uri.getScheme();

		String proxyHost = System.getProperty(scheme + ".proxyHost");
		String proxyPort = System.getProperty(scheme + ".proxyPort");
		String proxyUser = System.getProperty(scheme + ".proxyUser");
		String proxyPassword = System.getProperty(scheme + ".proxyPassword");

		if ((proxyHost != null) && (proxyPort != null) && (proxyUser != null) &&
			(proxyPassword != null)) {

			credentialsProvider.setCredentials(
				new AuthScope(proxyHost, Integer.parseInt(proxyPort)),
				new UsernamePasswordCredentials(proxyUser, proxyPassword));
		}

		httpClientBuilder.useSystemProperties();

		return httpClientBuilder;
	}

}