/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.object.internal.related.models;

import com.liferay.object.constants.ObjectRelationshipConstants;
import com.liferay.object.exception.RequiredObjectRelationshipException;
import com.liferay.object.model.ObjectDefinition;
import com.liferay.object.model.ObjectEntry;
import com.liferay.object.model.ObjectRelationship;
import com.liferay.object.related.models.ObjectRelatedModelsProvider;
import com.liferay.object.service.ObjectEntryService;
import com.liferay.object.service.ObjectRelationshipLocalService;
import com.liferay.petra.sql.dsl.expression.Predicate;
import com.liferay.portal.kernel.dao.orm.QueryUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.search.Sort;
import com.liferay.portal.kernel.service.ServiceContext;

import java.util.List;
import java.util.Objects;

/**
 * @author Marco Leo
 * @author Brian Wing Shun Chan
 */
public class ObjectEntryMtoMObjectRelatedModelsProviderImpl
	implements ObjectRelatedModelsProvider<ObjectEntry> {

	public ObjectEntryMtoMObjectRelatedModelsProviderImpl(
		ObjectDefinition objectDefinition,
		ObjectEntryService objectEntryService,
		ObjectRelationshipLocalService objectRelationshipLocalService) {

		_objectEntryService = objectEntryService;
		_objectRelationshipLocalService = objectRelationshipLocalService;

		_className = objectDefinition.getClassName();
		_companyId = objectDefinition.getCompanyId();
	}

	@Override
	public void deleteRelatedModel(
			long userId, long groupId, long objectRelationshipId,
			long primaryKey, String deletionType)
		throws PortalException {

		List<ObjectEntry> relatedModels = getRelatedModels(
			groupId, objectRelationshipId, null, primaryKey, null,
			QueryUtil.ALL_POS, QueryUtil.ALL_POS, null);

		if (relatedModels.isEmpty()) {
			return;
		}

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		if (Objects.equals(
				deletionType,
				ObjectRelationshipConstants.DELETION_TYPE_PREVENT) &&
			!objectRelationship.isReverse()) {

			throw new RequiredObjectRelationshipException(objectRelationship);
		}

		_objectRelationshipLocalService.
			deleteObjectRelationshipMappingTableValues(
				objectRelationshipId, primaryKey);

		if (Objects.equals(
				deletionType,
				ObjectRelationshipConstants.DELETION_TYPE_CASCADE) &&
			!objectRelationship.isReverse()) {

			for (ObjectEntry objectEntry : relatedModels) {
				_objectEntryService.deleteObjectEntry(
					objectEntry.getObjectEntryId());
			}
		}
	}

	@Override
	public void disassociateRelatedModels(
			long userId, long objectRelationshipId, long primaryKey1,
			long primaryKey2)
		throws PortalException {

		_objectRelationshipLocalService.
			deleteObjectRelationshipMappingTableValues(
				objectRelationshipId, primaryKey1, primaryKey2);
	}

	@Override
	public String getClassName() {
		return _className;
	}

	@Override
	public long getCompanyId() {
		return _companyId;
	}

	@Override
	public String getObjectRelationshipType() {
		return ObjectRelationshipConstants.TYPE_MANY_TO_MANY;
	}

	public List<ObjectEntry> getRelatedModels(
			long groupId, long objectRelationshipId, Predicate predicate,
			long primaryKey, String search, int start, int end, Sort[] sorts)
		throws PortalException {

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		return _objectEntryService.getManyToManyObjectEntries(
			groupId, objectRelationship.getObjectRelationshipId(), primaryKey,
			true, objectRelationship.isReverse(), search, start, end);
	}

	@Override
	public int getRelatedModelsCount(
			long groupId, long objectRelationshipId, Predicate predicate,
			long primaryKey, String search)
		throws PortalException {

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		return _objectEntryService.getManyToManyObjectEntriesCount(
			groupId, objectRelationship.getObjectRelationshipId(), primaryKey,
			true, objectRelationship.isReverse(), search);
	}

	@Override
	public List<ObjectEntry> getUnrelatedModels(
			long companyId, long groupId, ObjectDefinition objectDefinition,
			long objectEntryId, long objectRelationshipId, String search,
			int start, int end)
		throws PortalException {

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		return _objectEntryService.getManyToManyObjectEntries(
			groupId, objectRelationship.getObjectRelationshipId(),
			objectEntryId, false, objectRelationship.isReverse(), search, start,
			end);
	}

	@Override
	public int getUnrelatedModelsCount(
			long companyId, long groupId, ObjectDefinition objectDefinition,
			long objectEntryId, long objectRelationshipId, String search)
		throws PortalException {

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		return _objectEntryService.getManyToManyObjectEntriesCount(
			groupId, objectRelationship.getObjectRelationshipId(),
			objectEntryId, false, objectRelationship.isReverse(), search);
	}

	@Override
	public void moveRelatedModelToTrash(
			long userId, long groupId, long objectRelationshipId,
			long primaryKey, String deletionType)
		throws PortalException {

		List<ObjectEntry> relatedModels = getRelatedModels(
			groupId, objectRelationshipId, null, primaryKey, null,
			QueryUtil.ALL_POS, QueryUtil.ALL_POS, null);

		if (relatedModels.isEmpty()) {
			return;
		}

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		if (Objects.equals(
				deletionType,
				ObjectRelationshipConstants.DELETION_TYPE_PREVENT) &&
			!objectRelationship.isReverse()) {

			throw new RequiredObjectRelationshipException(objectRelationship);
		}

		if (Objects.equals(
				deletionType,
				ObjectRelationshipConstants.DELETION_TYPE_CASCADE) &&
			!objectRelationship.isReverse()) {

			for (ObjectEntry objectEntry : relatedModels) {
				_objectEntryService.moveObjectEntryToTrash(
					objectEntry, new ServiceContext());
			}
		}
	}

	@Override
	public void restoreRelatedModelsFromTrash(
			long userId, long groupId, long objectRelationshipId,
			long primaryKey, String deletionType)
		throws PortalException {

		ObjectRelationship objectRelationship =
			_objectRelationshipLocalService.getObjectRelationship(
				objectRelationshipId);

		if (!Objects.equals(
				deletionType,
				ObjectRelationshipConstants.DELETION_TYPE_CASCADE) ||
			objectRelationship.isReverse()) {

			return;
		}

		for (ObjectEntry objectEntry :
				getRelatedModels(
					groupId, objectRelationshipId, null, primaryKey, null,
					QueryUtil.ALL_POS, QueryUtil.ALL_POS, null)) {

			if (!objectEntry.isInTrash()) {
				continue;
			}

			_objectEntryService.restoreObjectEntryFromTrash(
				objectEntry, new ServiceContext());
		}
	}

	private final String _className;
	private final long _companyId;
	private final ObjectEntryService _objectEntryService;
	private final ObjectRelationshipLocalService
		_objectRelationshipLocalService;

}