/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.oauth.client.persistence.service.impl;

import com.liferay.oauth.client.persistence.constants.OAuthClientPersistenceActionKeys;
import com.liferay.oauth.client.persistence.model.OAuthClientASLocalMetadata;
import com.liferay.oauth.client.persistence.service.base.OAuthClientASLocalMetadataServiceBaseImpl;
import com.liferay.portal.aop.AopService;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.model.GroupConstants;
import com.liferay.portal.kernel.security.permission.ActionKeys;
import com.liferay.portal.kernel.security.permission.resource.ModelResourcePermission;
import com.liferay.portal.kernel.security.permission.resource.ModelResourcePermissionUtil;
import com.liferay.portal.kernel.util.OrderByComparator;

import java.util.List;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Arthur Chan
 */
@Component(
	property = {
		"json.web.service.context.name=oauthclient",
		"json.web.service.context.path=OAuthClientASLocalMetadata"
	},
	service = AopService.class
)
public class OAuthClientASLocalMetadataServiceImpl
	extends OAuthClientASLocalMetadataServiceBaseImpl {

	@Override
	public OAuthClientASLocalMetadata addOAuthClientASLocalMetadata(
			String metadataJSON, String wellKnownURISuffix)
		throws PortalException {

		ModelResourcePermissionUtil.check(
			_oAuthClientASLocalMetadataModelResourcePermission,
			getPermissionChecker(), GroupConstants.DEFAULT_LIVE_GROUP_ID, 0,
			OAuthClientPersistenceActionKeys.
				ACTION_ADD_OAUTH_CLIENT_AS_LOCAL_METADATA);

		return oAuthClientASLocalMetadataLocalService.
			addOAuthClientASLocalMetadata(
				getUserId(), metadataJSON, wellKnownURISuffix);
	}

	public OAuthClientASLocalMetadata addOAuthClientASLocalMetadata(
			String authorizationEndpoint, String issuer, String jwksURI,
			boolean localWellKnownEnabled, String[] supportedGrantTypes,
			String[] supportedScopes, String[] supportedSubjectTypes,
			String tokenEndpoint, String userInfoEndpoint)
		throws PortalException {

		ModelResourcePermissionUtil.check(
			_oAuthClientASLocalMetadataModelResourcePermission,
			getPermissionChecker(), GroupConstants.DEFAULT_LIVE_GROUP_ID, 0,
			OAuthClientPersistenceActionKeys.
				ACTION_ADD_OAUTH_CLIENT_AS_LOCAL_METADATA);

		return oAuthClientASLocalMetadataLocalService.
			addOAuthClientASLocalMetadata(
				getUserId(), authorizationEndpoint, issuer, jwksURI,
				localWellKnownEnabled, supportedGrantTypes, supportedScopes,
				supportedSubjectTypes, tokenEndpoint, userInfoEndpoint);
	}

	@Override
	public OAuthClientASLocalMetadata deleteOAuthClientASLocalMetadata(
			long oAuthClientASLocalMetadataId)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataLocalService.
				getOAuthClientASLocalMetadata(oAuthClientASLocalMetadataId);

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(), oAuthClientASLocalMetadata,
			ActionKeys.DELETE);

		return oAuthClientASLocalMetadataLocalService.
			deleteOAuthClientASLocalMetadata(oAuthClientASLocalMetadata);
	}

	@Override
	public OAuthClientASLocalMetadata deleteOAuthClientASLocalMetadata(
			String localWellKnownURI)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataLocalService.
				getOAuthClientASLocalMetadata(localWellKnownURI);

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(), oAuthClientASLocalMetadata,
			ActionKeys.DELETE);

		return oAuthClientASLocalMetadataLocalService.
			deleteOAuthClientASLocalMetadata(oAuthClientASLocalMetadata);
	}

	@Override
	public OAuthClientASLocalMetadata fetchOAuthClientASLocalMetadata(
			long oAuthClientASLocalMetadataId)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataPersistence.fetchByPrimaryKey(
				oAuthClientASLocalMetadataId);

		if (oAuthClientASLocalMetadata != null) {
			_oAuthClientASLocalMetadataModelResourcePermission.check(
				getPermissionChecker(), oAuthClientASLocalMetadata,
				ActionKeys.VIEW);
		}

		return oAuthClientASLocalMetadata;
	}

	@Override
	public OAuthClientASLocalMetadata fetchOAuthClientASLocalMetadata(
			long companyId, String issuer)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataPersistence.fetchByC_I(companyId, issuer);

		if (oAuthClientASLocalMetadata != null) {
			_oAuthClientASLocalMetadataModelResourcePermission.check(
				getPermissionChecker(), oAuthClientASLocalMetadata,
				ActionKeys.VIEW);
		}

		return oAuthClientASLocalMetadata;
	}

	@Override
	public List<OAuthClientASLocalMetadata>
		getCompanyOAuthClientASLocalMetadata(long companyId) {

		return oAuthClientASLocalMetadataPersistence.filterFindByCompanyId(
			companyId);
	}

	@Override
	public List<OAuthClientASLocalMetadata>
		getCompanyOAuthClientASLocalMetadata(
			long companyId, int start, int end) {

		return oAuthClientASLocalMetadataPersistence.filterFindByCompanyId(
			companyId, start, end);
	}

	@Override
	public OAuthClientASLocalMetadata getOAuthClientASLocalMetadata(
			long companyId, boolean localWellKnownEnabled,
			OrderByComparator<OAuthClientASLocalMetadata> orderByComparator)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataPersistence.findByC_L_First(
				companyId, localWellKnownEnabled, orderByComparator);

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(), oAuthClientASLocalMetadata,
			ActionKeys.VIEW);

		return oAuthClientASLocalMetadata;
	}

	@Override
	public OAuthClientASLocalMetadata getOAuthClientASLocalMetadata(
			long companyId, String issuer)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataPersistence.findByC_I(companyId, issuer);

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(), oAuthClientASLocalMetadata,
			ActionKeys.VIEW);

		return oAuthClientASLocalMetadata;
	}

	@Override
	public OAuthClientASLocalMetadata getOAuthClientASLocalMetadata(
			String localWellKnownURI)
		throws PortalException {

		OAuthClientASLocalMetadata oAuthClientASLocalMetadata =
			oAuthClientASLocalMetadataPersistence.findByLocalWellKnownURI(
				localWellKnownURI);

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(), oAuthClientASLocalMetadata,
			ActionKeys.VIEW);

		return oAuthClientASLocalMetadata;
	}

	@Override
	public List<OAuthClientASLocalMetadata> getUserOAuthClientASLocalMetadata(
		long userId) {

		return oAuthClientASLocalMetadataPersistence.filterFindByUserId(userId);
	}

	@Override
	public List<OAuthClientASLocalMetadata> getUserOAuthClientASLocalMetadata(
		long userId, int start, int end) {

		return oAuthClientASLocalMetadataPersistence.filterFindByUserId(
			userId, start, end);
	}

	@Override
	public OAuthClientASLocalMetadata updateOAuthClientASLocalMetadata(
			long oAuthClientASLocalMetadataId, String metadataJSON,
			String wellKnownURISuffix)
		throws PortalException {

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(),
			oAuthClientASLocalMetadataPersistence.findByPrimaryKey(
				oAuthClientASLocalMetadataId),
			ActionKeys.UPDATE);

		return oAuthClientASLocalMetadataLocalService.
			updateOAuthClientASLocalMetadata(
				oAuthClientASLocalMetadataId, metadataJSON, wellKnownURISuffix);
	}

	public OAuthClientASLocalMetadata updateOAuthClientASLocalMetadata(
			long oAuthClientASLocalMetadataId, String authorizationEndpoint,
			String issuer, String jwksURI, boolean localWellKnownEnabled,
			String[] supportedGrantTypes, String[] supportedScopes,
			String[] supportedSubjectTypes, String tokenEndpoint,
			String userInfoEndpoint)
		throws PortalException {

		_oAuthClientASLocalMetadataModelResourcePermission.check(
			getPermissionChecker(),
			oAuthClientASLocalMetadataPersistence.findByPrimaryKey(
				oAuthClientASLocalMetadataId),
			ActionKeys.UPDATE);

		return oAuthClientASLocalMetadataLocalService.
			updateOAuthClientASLocalMetadata(
				oAuthClientASLocalMetadataId, authorizationEndpoint, issuer,
				jwksURI, localWellKnownEnabled, supportedGrantTypes,
				supportedScopes, supportedSubjectTypes, tokenEndpoint,
				userInfoEndpoint);
	}

	@Reference(
		target = "(model.class.name=com.liferay.oauth.client.persistence.model.OAuthClientASLocalMetadata)"
	)
	private ModelResourcePermission<OAuthClientASLocalMetadata>
		_oAuthClientASLocalMetadataModelResourcePermission;

}