/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.mobile.device.rules.internal.events;

import com.liferay.mobile.device.rules.action.ActionHandlerManagerUtil;
import com.liferay.mobile.device.rules.internal.MDRRuleGroupInstanceImpl;
import com.liferay.mobile.device.rules.model.MDRAction;
import com.liferay.mobile.device.rules.model.MDRRuleGroupInstance;
import com.liferay.mobile.device.rules.rule.RuleGroupProcessorUtil;
import com.liferay.mobile.device.rules.service.MDRActionLocalService;
import com.liferay.portal.kernel.events.Action;
import com.liferay.portal.kernel.events.LifecycleAction;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.mobile.device.Device;
import com.liferay.portal.kernel.mobile.device.DeviceDetectionUtil;
import com.liferay.portal.kernel.mobile.device.UnknownDevice;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.TransientValue;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portal.util.PropsValues;

import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Edward Han
 */
@Component(
	property = "key=servlet.service.events.pre", service = LifecycleAction.class
)
public class MDRServicePreAction extends Action {

	@Override
	public void run(HttpServletRequest request, HttpServletResponse response) {
		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		Device device = null;

		if (PropsValues.MOBILE_DEVICE_SESSION_CACHE_ENABLED) {
			HttpSession session = request.getSession();

			TransientValue<Device> transientValue =
				(TransientValue<Device>)session.getAttribute(WebKeys.DEVICE);

			if (transientValue != null) {
				device = transientValue.getValue();
			}

			if (device == null) {
				device = DeviceDetectionUtil.detectDevice(request);

				session.setAttribute(
					WebKeys.DEVICE, new TransientValue<>(device));
			}
		}
		else {
			device = DeviceDetectionUtil.detectDevice(request);
		}

		themeDisplay.setDevice(device);

		UnknownDevice unknownDevice = UnknownDevice.getInstance();

		if (device.equals(unknownDevice)) {
			return;
		}

		MDRRuleGroupInstance mdrRuleGroupInstance = null;

		try {
			mdrRuleGroupInstance = RuleGroupProcessorUtil.evaluateRuleGroups(
				themeDisplay);

			if (_log.isDebugEnabled()) {
				String logMessage =
					"Rule group evaluation returned rule group instance ";

				if (mdrRuleGroupInstance != null) {
					logMessage += mdrRuleGroupInstance.getRuleGroupInstanceId();
				}
				else {
					logMessage += "null";
				}

				_log.debug(logMessage);
			}
		}
		catch (Exception e) {
			if (_log.isWarnEnabled()) {
				_log.warn("Unable to retrieve rule group", e);
			}

			return;
		}

		if (mdrRuleGroupInstance == null) {
			return;
		}

		themeDisplay.setMDRRuleGroupInstance(
			new MDRRuleGroupInstanceImpl(mdrRuleGroupInstance));

		try {
			List<MDRAction> mdrActions = _mdrActionLocalService.getActions(
				mdrRuleGroupInstance.getRuleGroupInstanceId());

			ActionHandlerManagerUtil.applyActions(
				mdrActions, request, response);
		}
		catch (Exception e) {
			if (_log.isWarnEnabled()) {
				_log.warn("Unable to apply device profile", e);
			}
		}
	}

	@Reference(unbind = "-")
	protected void setMDRActionLocalService(
		MDRActionLocalService mdrActionLocalService) {

		_mdrActionLocalService = mdrActionLocalService;
	}

	private static final Log _log = LogFactoryUtil.getLog(
		MDRServicePreAction.class);

	private MDRActionLocalService _mdrActionLocalService;

}