/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.mail.reader.service.impl;

import com.liferay.document.library.kernel.store.DLStoreUtil;
import com.liferay.mail.reader.model.Attachment;
import com.liferay.mail.reader.model.Message;
import com.liferay.mail.reader.service.base.AttachmentLocalServiceBaseImpl;
import com.liferay.petra.string.StringBundler;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.aop.AopService;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.exception.SystemException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.CompanyConstants;
import com.liferay.portal.kernel.model.User;
import com.liferay.portal.kernel.util.FileUtil;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import java.util.List;

import org.osgi.service.component.annotations.Component;

/**
 * @author Scott Lee
 */
@Component(
	property = "model.class.name=com.liferay.mail.reader.model.Attachment",
	service = AopService.class
)
public class AttachmentLocalServiceImpl extends AttachmentLocalServiceBaseImpl {

	@Override
	public Attachment addAttachment(
			long userId, long messageId, String contentPath, String fileName,
			long size, File file)
		throws PortalException {

		// Attachment

		User user = userLocalService.getUser(userId);
		Message message = messagePersistence.findByPrimaryKey(messageId);

		long attachmentId = counterLocalService.increment();

		Attachment attachment = attachmentPersistence.create(attachmentId);

		attachment.setCompanyId(user.getCompanyId());
		attachment.setUserId(user.getUserId());
		attachment.setAccountId(message.getAccountId());
		attachment.setFolderId(message.getFolderId());
		attachment.setMessageId(messageId);
		attachment.setContentPath(contentPath);
		attachment.setFileName(fileName);
		attachment.setSize(size);

		attachment = attachmentPersistence.update(attachment);

		// File

		if (file != null) {
			if (!file.exists()) {
				throw new PortalException(new FileNotFoundException());
			}

			String filePath = getFilePath(attachment.getMessageId(), fileName);

			try {
				DLStoreUtil.addFile(
					attachment.getCompanyId(), _REPOSITORY_ID, filePath, file);
			}
			catch (PortalException portalException) {
				if (_log.isDebugEnabled()) {
					_log.debug(portalException, portalException);
				}
			}
		}

		return attachment;
	}

	@Override
	public Attachment deleteAttachment(long attachmentId)
		throws PortalException {

		// Attachment

		Attachment attachment = attachmentPersistence.findByPrimaryKey(
			attachmentId);

		attachmentPersistence.remove(attachmentId);

		// File

		String filePath = getFilePath(
			attachment.getMessageId(), attachment.getFileName());

		DLStoreUtil.deleteFile(
			attachment.getCompanyId(), _REPOSITORY_ID, filePath);

		return attachment;
	}

	@Override
	public void deleteAttachments(long companyId, long messageId)
		throws PortalException {

		// Attachments

		List<Attachment> attachments = attachmentPersistence.findByMessageId(
			messageId);

		for (Attachment attachment : attachments) {
			deleteAttachment(attachment);
		}

		// File

		DLStoreUtil.deleteDirectory(
			companyId, _REPOSITORY_ID, getDirectoryPath(messageId));
	}

	@Override
	public List<Attachment> getAttachments(long messageId) {
		return attachmentPersistence.findByMessageId(messageId);
	}

	@Override
	public File getFile(long attachmentId) throws PortalException {
		try {
			File file = FileUtil.createTempFile();

			FileUtil.write(file, getInputStream(attachmentId));

			return file;
		}
		catch (IOException ioException) {
			throw new SystemException(ioException);
		}
	}

	@Override
	public InputStream getInputStream(long attachmentId)
		throws PortalException {

		Attachment attachment = attachmentPersistence.findByPrimaryKey(
			attachmentId);

		String filePath = getFilePath(
			attachment.getMessageId(), attachment.getFileName());

		return DLStoreUtil.getFileAsStream(
			attachment.getCompanyId(), _REPOSITORY_ID, filePath);
	}

	protected String getDirectoryPath(long messageId) {
		return _DIRECTORY_PATH_PREFIX.concat(String.valueOf(messageId));
	}

	protected String getFilePath(long messageId, String fileName) {
		return StringBundler.concat(
			getDirectoryPath(messageId), StringPool.SLASH, fileName);
	}

	private static final String _DIRECTORY_PATH_PREFIX = "mail/";

	private static final long _REPOSITORY_ID = CompanyConstants.SYSTEM;

	private static final Log _log = LogFactoryUtil.getLog(
		AttachmentLocalServiceImpl.class);

}