/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

import {debounce} from 'frontend-js-web';

import {convertToFormData, makeFetch} from './fetch.es';
import {PagesVisitor} from './visitors.es';

const EVALUATOR_URL =
	themeDisplay.getPathContext() +
	'/o/dynamic-data-mapping-form-context-provider/';

let controller = null;

export const mergeFieldOptions = (field, newField) => {
	let newValue = {...newField.value};

	Object.keys(newValue).forEach((languageId) => {
		newValue = {
			...newValue,
			[languageId]: newValue[languageId].map((option) => {
				const existingOption =
					field.value &&
					field.value[languageId] &&
					field.value[languageId].find(
						({value}) => value === option.value
					);

				return {
					...option,
					edited: existingOption && existingOption.edited,
				};
			}),
		};
	});

	return newValue;
};

export const mergePages = (
	defaultLanguageId,
	editingLanguageId,
	fieldName,
	newPages,
	sourcePages,
	viewMode
) => {
	const newPagesVisitor = new PagesVisitor(newPages);
	const sourcePagesVisitor = new PagesVisitor(sourcePages);

	return newPagesVisitor.mapFields(
		(field) => {
			let sourceField =
				sourcePagesVisitor.findField(({name}) => name === field.name) ||
				{};

			if (!sourceField || Object.keys(sourceField).length === 0) {
				sourceField =
					sourcePagesVisitor.findField(
						({fieldName}) => fieldName === field.fieldName
					) || {};
			}

			let fieldValue = field.valueChanged
				? field.value
				: sourceField.value;

			if (
				field.visible !== sourceField.visible &&
				field.visible &&
				viewMode
			) {
				fieldValue = '';
			}

			let newField = {
				...sourceField,
				...field,
				defaultLanguageId,
				displayErrors:
					sourceField.displayErrors || field.fieldName === fieldName,
				editingLanguageId,
				valid: field.valid !== false,
				value: fieldValue,
			};

			if (newField.type === 'options') {
				newField = {
					...newField,
					value: mergeFieldOptions(sourceField, newField),
				};
			}

			if (newField.localizable) {
				newField = {
					...newField,
					localizedValue: {
						...sourceField.localizedValue,
					},
				};
			}

			return newField;
		},
		false,
		true
	);
};

const doEvaluate = debounce((fieldName, evaluatorContext, callback) => {
	const {
		defaultLanguageId,
		editingLanguageId,
		groupId,
		pages,
		portletNamespace,
		viewMode,
	} = evaluatorContext;

	if (controller) {
		controller.abort();
	}

	if (window.AbortController) {
		controller = new AbortController();
	}

	makeFetch({
		body: convertToFormData({
			languageId: editingLanguageId,
			p_auth: Liferay.authToken,
			portletNamespace,
			serializedFormContext: JSON.stringify({
				...evaluatorContext,
				groupId: groupId ? groupId : themeDisplay.getScopeGroupId(),
				portletNamespace,
			}),
			trigger: fieldName,
		}),
		signal: controller && controller.signal,
		url: EVALUATOR_URL,
	})
		.then((newPages) => {
			const mergedPages = mergePages(
				defaultLanguageId,
				editingLanguageId,
				fieldName,
				newPages,
				pages,
				viewMode
			);

			callback(null, mergedPages);
		})
		.catch((error) => callback(error));
}, 600);

export const evaluate = (fieldName, evaluatorContext) => {
	return new Promise((resolve, reject) => {
		doEvaluate(fieldName, evaluatorContext, (error, pages) => {
			if (error) {
				return reject(error);
			}

			resolve(pages);
		});
	});
};
