/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.batch.engine.internal.installer;

import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.IOException;
import java.io.InputStream;

import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * @author Igor Beslic
 */
public class ClassicBatchEngineZipUnitImpl<T> implements BatchEngineZipUnit<T> {

	public ClassicBatchEngineZipUnitImpl(
		ZipFile zipFile, ZipEntry... zipEntries) {

		_zipFile = zipFile;

		if ((zipEntries == null) || (zipEntries.length > 2)) {
			return;
		}

		for (ZipEntry zipEntry : zipEntries) {
			if (_isBatchEngineConfiguration(zipEntry.getName())) {
				_configurationZipEntry = zipEntry;

				continue;
			}

			_dataZipEntry = zipEntry;
		}
	}

	@Override
	public T getBatchEngineConfiguration(Class<T> clazz) throws IOException {
		try (InputStream inputStream = _zipFile.getInputStream(
				_configurationZipEntry)) {

			ObjectMapper objectMapper = new ObjectMapper();

			return objectMapper.readValue(inputStream, clazz);
		}
	}

	@Override
	public InputStream getConfigurationInputStream() throws IOException {
		return _zipFile.getInputStream(_configurationZipEntry);
	}

	@Override
	public String getDataFileName() {
		return _dataZipEntry.getName();
	}

	@Override
	public InputStream getDataInputStream() throws IOException {
		return _zipFile.getInputStream(_dataZipEntry);
	}

	@Override
	public String getZipFileName() {
		return _zipFile.getName();
	}

	public boolean isValid() {
		if ((_configurationZipEntry == null) || (_dataZipEntry == null)) {
			return false;
		}

		return true;
	}

	private boolean _isBatchEngineConfiguration(String zipEntryName) {
		if (Objects.equals(zipEntryName, "batch-engine.json") ||
			zipEntryName.endsWith("/batch-engine.json")) {

			return true;
		}

		return false;
	}

	private ZipEntry _configurationZipEntry;
	private ZipEntry _dataZipEntry;
	private final ZipFile _zipFile;

}