/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.batch.engine.internal.writer;

import com.liferay.petra.function.UnsafeFunction;
import com.liferay.petra.string.CharPool;
import com.liferay.petra.string.StringPool;

import java.lang.reflect.Field;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author Shuyang Zhou
 * @author Igor Beslic
 */
public class ColumnValuesExtractor {

	public ColumnValuesExtractor(
		Map<String, Field> fieldMap, List<String> fieldNames) {

		List<UnsafeFunction<Object, Object, ReflectiveOperationException>>
			unsafeFunctions = new ArrayList<>(fieldNames.size());

		for (String fieldName : fieldNames) {
			Field field = fieldMap.get(fieldName);

			if (field != null) {
				if (ItemClassIndexUtil.isSingleColumnAdoptableValue(
						field.getType())) {

					unsafeFunctions.add(
						item -> {
							if (field.get(item) == null) {
								return StringPool.BLANK;
							}

							return field.get(item);
						});

					continue;
				}

				if (ItemClassIndexUtil.isSingleColumnAdoptableArray(
						field.getType())) {

					unsafeFunctions.add(item -> StringPool.BLANK);

					continue;
				}
			}

			int index = fieldName.indexOf(CharPool.UNDERLINE);

			if (index == -1) {
				throw new IllegalArgumentException(
					"Invalid field name : " + fieldName);
			}

			String prefixFieldName = fieldName.substring(0, index);

			Field mapField = fieldMap.get(prefixFieldName);

			if (mapField == null) {
				throw new IllegalArgumentException(
					"Invalid field name : " + fieldName);
			}

			if (mapField.getType() != Map.class) {
				throw new IllegalArgumentException(
					"Invalid field name : " + fieldName +
						", it is not Map type.");
			}

			String key = fieldName.substring(index + 1);

			unsafeFunctions.add(
				item -> {
					Map<?, ?> map = (Map<?, ?>)mapField.get(item);

					Object value = map.get(key);

					if (value == null) {
						return StringPool.BLANK;
					}

					return value;
				});
		}

		_unsafeFunctions = unsafeFunctions;
	}

	public List<Object> extractValues(Object item)
		throws ReflectiveOperationException {

		List<Object> values = new ArrayList<>(_unsafeFunctions.size());

		for (UnsafeFunction<Object, Object, ReflectiveOperationException>
				unsafeFunction : _unsafeFunctions) {

			values.add(unsafeFunction.apply(item));
		}

		return values;
	}

	private final List
		<UnsafeFunction<Object, Object, ReflectiveOperationException>>
			_unsafeFunctions;

}