/**
 * SPDX-FileCopyrightText: (c) 2023 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.blade.cli;

import com.liferay.blade.cli.util.CombinedClassLoader;
import com.liferay.blade.cli.util.FileUtil;
import com.liferay.blade.cli.util.ProcessesUtil;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

import java.util.HashSet;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Christopher Bryan Boyd
 */
public class ExtensionsClassLoaderSupplier implements AutoCloseable, Supplier<ClassLoader> {

	public ExtensionsClassLoaderSupplier(Path extensionsPath) {
		_extensionsPath = extensionsPath;
	}

	@Override
	public void close() throws Exception {
		if ((_serviceLoaderClassLoader != null) && (_serviceLoaderClassLoader instanceof Closeable)) {
			Closeable closeable = (Closeable)_serviceLoaderClassLoader;

			closeable.close();
		}

		if ((_tempExtensionsDirectory != null) && Files.exists(_tempExtensionsDirectory)) {
			try {
				FileUtil.deleteDirIfExists(_tempExtensionsDirectory);
			}
			catch (IOException ioException) {
			}
		}
	}

	@Override
	public ClassLoader get() {
		try {
			if (_serviceLoaderClassLoader == null) {
				long pid = ProcessesUtil.getCurrentProcessId();

				_tempExtensionsDirectory = Files.createTempDirectory("blade-extensions-" + pid + "-");

				FileUtil.copyDir(_extensionsPath, _tempExtensionsDirectory);

				_extractBladeExtensions(_tempExtensionsDirectory);

				URL[] jarUrls = _getJarUrls(_tempExtensionsDirectory);

				Thread thread = Thread.currentThread();

				ClassLoader currentClassLoader = thread.getContextClassLoader();

				ClassLoader urlClassLoader = new URLClassLoader(jarUrls, getClass().getClassLoader());

				_serviceLoaderClassLoader = new CombinedClassLoader(currentClassLoader, urlClassLoader);
			}

			return _serviceLoaderClassLoader;
		}
		catch (Throwable throwable) {
			throw new RuntimeException(throwable);
		}
	}

	private static URL _convertUriToUrl(URI uri) {
		try {
			return uri.toURL();
		}
		catch (MalformedURLException malformedURLException) {
		}

		return null;
	}

	private void _extractBladeExtensions(Path extensionsDirectory) throws IOException {
		try (InputStream inputStream = Extensions.class.getResourceAsStream("/blade-extensions-versions.properties")) {
			if (inputStream == null) {
				return;
			}

			Properties properties = new Properties();

			properties.load(inputStream);

			Set<Object> keySet = properties.keySet();

			ClassLoader classLoader = Extensions.class.getClassLoader();

			try {
				Set<String> extensions = new HashSet<>();

				for (Object key : keySet) {
					String extension = key.toString() + "-" + properties.getProperty(key.toString()) + ".jar";

					if (!extension.startsWith("com.liferay.project.templates")) {
						if (classLoader.getResource(extension) != null) {
							extensions.add(extension);
						}
						else {
							System.err.println("Warning: Unable to locate " + extension);
						}
					}
				}

				for (String extension : extensions) {
					try (InputStream extensionInputStream = classLoader.getResourceAsStream(extension)) {
						Path extensionPath = extensionsDirectory.resolve(extension);

						Files.copy(extensionInputStream, extensionPath, StandardCopyOption.REPLACE_EXISTING);
					}
					catch (Throwable throwable) {
						StringBuilder sb = new StringBuilder();

						sb.append("Error encountered while loading custom extensions.");
						sb.append(System.lineSeparator());
						sb.append(throwable.getMessage());
						sb.append(System.lineSeparator());
						sb.append("Not loading extension " + extension + ".");
						sb.append(System.lineSeparator());

						String errorString = sb.toString();

						System.err.println(errorString);
					}
				}
			}
			catch (NoSuchElementException noSuchElementException) {
				StringBuilder sb = new StringBuilder();

				sb.append("Error encountered while loading custom extensions.");
				sb.append(System.lineSeparator());
				sb.append(noSuchElementException.getMessage());
				sb.append(System.lineSeparator());
				sb.append("Only built-in commands will be recognized.");
				sb.append(System.lineSeparator());

				String errorString = sb.toString();

				System.err.println(errorString);
			}
			catch (Throwable throwable) {
				String errorMessage = "Error encountered while loading custom extensions." + System.lineSeparator();

				throw new RuntimeException(errorMessage, throwable);
			}
		}
	}

	private URL[] _getJarUrls(Path jarsPath) throws IOException {
		try (Stream<Path> files = Files.list(jarsPath)) {
			return files.filter(
				path -> {
					String file = path.toString();

					return file.endsWith(".jar");
				}
			).map(
				Path::toUri
			).map(
				ExtensionsClassLoaderSupplier::_convertUriToUrl
			).filter(
				url -> url != null
			).collect(
				Collectors.toSet()
			).toArray(
				new URL[0]
			);
		}
	}

	private final Path _extensionsPath;
	private ClassLoader _serviceLoaderClassLoader = null;
	private Path _tempExtensionsDirectory;

}