package com.leiainc.androidsdk.display;

import android.content.Context;
import android.content.res.Resources;
import android.util.DisplayMetrics;
import com.leia.android.lights.DisplayConfig;
import java.util.List;

/**
 * Convenience wrapper that adds: - orientation handling - Default values for when DisplayManager is
 * null. - Marshalling data into arrays.
 */
@SuppressWarnings("unused")
public class SimpleDisplayQuery {
    private final LeiaDisplayManagerSource mDisplayManager;
    private final boolean mIsPortraitNativeOrientation;

    public SimpleDisplayQuery(Context context) {
        mDisplayManager = LeiaSDK.getDisplayManager(context);
        mIsPortraitNativeOrientation = isPortraitNativeOrientation();
    }

    /**
     * @return The view sharpening coefficients of the display.
     *     <p>View sharpening is a sharpening algorithm specifically designed to make sure the
     *     end-user's eyes are getting the most crisp, beautiful image possible.
     *     <p>Please see the SDK documentation for information on how to implement view sharpening.
     */
    public float[] getViewSharpening() {
        return getViewSharpening(isCurrentlyInPortraitOrientation());
    }

    public float[] getViewSharpening(boolean is_portrait) {
        if (mDisplayManager == null) {
            return new float[] {0.0f, 0.0f};
        }
        DisplayConfig c = mDisplayManager.getDisplayConfig();
        List<Float> coefficients;
        if (doesCurrentOrientationMatchNativeOrientation(is_portrait)) {
            coefficients = c.getViewSharpeningCoefficients().x;
        } else {
            coefficients = c.getViewSharpeningCoefficients().y;
        }
        // TODO: I think this can be replaced by coefficients.toArray(new float[])
        int size = coefficients.size();
        float[] params = new float[size];
        for (int i = 0; i < size; ++i) {
            params[i] = coefficients.get(i);
        }
        return params;
    }

    /**
     * @return The alignment offset of the display.
     *     <p>This value is the offset in pixels from center alignment.
     */
    public float getAlignmentOffset() {
        return getAlignmentOffset(isCurrentlyInPortraitOrientation());
    }

    public float getAlignmentOffset(boolean is_portrait) {
        if (mDisplayManager == null) {
            return 0.0f;
        }

        DisplayConfig c = mDisplayManager.getDisplayConfig();
        if (doesCurrentOrientationMatchNativeOrientation(is_portrait)) {
            return c.getAlignmentOffset().x;
        } else {
            return c.getAlignmentOffset().y;
        }
    }

    /**
     * @return The system disparity.
     *     <p>This is defined as the disparity between pixels when rendering content at infinity.
     */
    public int getSystemDisparity() {
        if (mDisplayManager == null) {
            return 0;
        }
        DisplayConfig c = mDisplayManager.getDisplayConfig();
        return c.getSystemDisparity();
    }

    /** @return The resolution of the display in pixels. */
    public int[] getScreenResolution() {
        return getScreenResolution(isCurrentlyInPortraitOrientation());
    }

    public int[] getScreenResolution(boolean is_portrait) {
        int[] resolution = {0, 0};
        if (mDisplayManager == null) {
            return resolution;
        }
        DisplayConfig c = mDisplayManager.getDisplayConfig();
        if (doesCurrentOrientationMatchNativeOrientation(is_portrait)) {
            resolution[0] = c.getPanelResolution().x;
            resolution[1] = c.getPanelResolution().y;
        } else {
            resolution[1] = c.getPanelResolution().x;
            resolution[0] = c.getPanelResolution().y;
        }
        return resolution;
    }

    /** @return The number of multi-view views. */
    public int[] getNumAvailableViews() {
        return getNumAvailableViews(isCurrentlyInPortraitOrientation());
    }

    public int[] getNumAvailableViews(boolean is_portrait) {
        int[] dims = {0, 0};
        if (mDisplayManager == null) {
            return dims;
        }

        DisplayConfig c = mDisplayManager.getDisplayConfig();
        if (doesCurrentOrientationMatchNativeOrientation(is_portrait)) {
            dims[0] = c.getNumViews().x;
            dims[1] = c.getNumViews().y;
        } else {
            dims[1] = c.getNumViews().x;
            dims[0] = c.getNumViews().y;
        }
        return dims;
    }

    /** @return The resolution per view, in pixels. */
    public int[] getViewResolution() {
        return getViewResolution(isCurrentlyInPortraitOrientation());
    }

    public int[] getViewResolution(boolean is_portrait) {
        int[] dims_view = {0, 0};
        if (mDisplayManager == null) {
            return dims_view;
        }

        DisplayConfig c = mDisplayManager.getDisplayConfig();
        if (doesCurrentOrientationMatchNativeOrientation(is_portrait)) {
            dims_view[0] = c.getViewResolution().x;
            dims_view[1] = c.getViewResolution().y;
        } else {
            dims_view[1] = c.getViewResolution().x;
            dims_view[0] = c.getViewResolution().y;
        }
        return dims_view;
    }

    public boolean doesCurrentOrientationMatchNativeOrientation(boolean is_portrait) {
        return is_portrait == mIsPortraitNativeOrientation;
    }

    public boolean isPortraitNativeOrientation() {
        boolean is_portrait_native_orientation = false;
        if (mDisplayManager != null) {
            DisplayConfig c = mDisplayManager.getDisplayConfig();
            DisplayConfig.XyPair<Integer> screen_resolution = c.getPanelResolution();
            if (screen_resolution.x < screen_resolution.y) {
                is_portrait_native_orientation = true;
            }
        }
        return is_portrait_native_orientation;
    }

    public boolean isCurrentlyInPortraitOrientation() {
        DisplayMetrics dm = Resources.getSystem().getDisplayMetrics();
        int width = dm.widthPixels;
        int height = dm.heightPixels;
        return height > width;
    }
}
