package com.leia.android.lights;

import com.leiainc.androidsdk.display.config.ClockwiseOrientation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A description of the Leia display's physical properties.
 *
 * <p>These can be used by an application to render multi-view content appropriate for the display.
 */
public class DisplayConfig {
    XyPair<Float> mDotPitchInMm;
    XyPair<Integer> mPanelResolution;
    XyPair<Integer> mNumViews;
    XyPair<Float> mAlignmentOffset;
    XyPair<List<Float>> mViewSharpeningCoefficients;
    int mSystemDisparity;
    Map<ClockwiseOrientation, int[][]> mInterlacingConfigs = new HashMap<>(0);

    /**
     * A pair of x-y coordinates.
     *
     * <p>IMPORTANT: The orientation of x and y are defined in terms of the device's "natural
     * orientation": In technical terms, x is defined as the direction of the RGB pixel stride.
     * Intuitively: phones are naturally in portrait orientation (x is the shorter side). tablets
     * are naturally in landscape orientation (x is the longer side).
     */
    public static class XyPair<T> {
        public final T x;
        public final T y;

        public XyPair(T x, T y) {
            this.x = x;
            this.y = y;
        }

        @Override
        public String toString() {
            return "{" + "x=" + arrayToStringHelper(x) + ", y=" + arrayToStringHelper(y) + '}';
        }

        private String arrayToStringHelper(T a) {
            if (a instanceof float[]) {
                return Arrays.toString((float[]) a);
            } else {
                return a.toString();
            }
        }
    }

    // Package-private constructor (only allowed to be instantiated by this package).
    DisplayConfig() {}

    /**
     * @return The dot pitch in millimeters.
     *     <p>This is the physical distance between pixels.
     */
    public XyPair<Float> getDotPitchInMm() {
        return mDotPitchInMm;
    }

    /** @return The resolution of the display in pixels. */
    public XyPair<Integer> getPanelResolution() {
        return mPanelResolution;
    }

    /** @return The number of multi-view views. */
    public XyPair<Integer> getNumViews() {
        return mNumViews;
    }

    /**
     * @return The alignment offset of the display.
     *     <p>In an idealized display, each diffractive backlight site should be directly under an
     *     LCD pixel site. This is not always realizable due to manufacturing tolerances.
     *     <p>This value is the offset in pixels from ideal alignment.
     */
    public XyPair<Float> getAlignmentOffset() {
        return mAlignmentOffset;
    }

    /**
     * @return The view sharpening coefficients of the display.
     *     <p>Developers should generally prefer the getViewSharpeningKernel() method.
     */
    public XyPair<List<Float>> getViewSharpeningCoefficients() {
        return mViewSharpeningCoefficients;
    }

    /**
     * @return The system disparity. This is defined as the disparity between pixels when rendering
     *     content at infinity.
     */
    public int getSystemDisparity() {
        return mSystemDisparity;
    }

    /**
     * @return The display's physical dimensions in Mm.
     *     <p>This value is derived from the panel resolution and dot pitch.
     */
    public XyPair<Integer> getDisplaySizeInMm() {
        int xDisplaySize = (int) (getPanelResolution().x * getDotPitchInMm().x);
        int yDisplaySize = (int) (getPanelResolution().y * getDotPitchInMm().y);
        return new XyPair<>(xDisplaySize, yDisplaySize);
    }

    /**
     * @return The resolution per view, in pixels.
     *     <p>This is derived from the panel resolution and the number of views.
     */
    public XyPair<Integer> getViewResolution() {
        int xViewResolution = getPanelResolution().x / getNumViews().x;
        int yViewResolution = getPanelResolution().y / getNumViews().y;
        return new XyPair<>(xViewResolution, yViewResolution);
    }

    /**
     * @return The view sharpening kernel.
     *     <p>This should be convolved with an interlaced image in order to reduce crosstalk between
     *     views. Note that the interlaced image must be represented in a linear color space.
     *     <p>This is currently a pair of length 5 vectors, but we may use a different length in the
     *     future.
     *     <p>This is derived from the view sharpening coefficients.
     */
    public XyPair<List<Float>> getViewSharpeningKernel() {
        List<Float> xKernel =
                viewSharpeningCoefficientsToKernel(
                        mViewSharpeningCoefficients.x.get(0), mViewSharpeningCoefficients.x.get(1));
        List<Float> yKernel =
                viewSharpeningCoefficientsToKernel(
                        mViewSharpeningCoefficients.y.get(0), mViewSharpeningCoefficients.y.get(1));
        return new XyPair<>(xKernel, yKernel);
    }

    public Map<ClockwiseOrientation, int[][]> getInterlacingConfigs() {
        return mInterlacingConfigs;
    }

    /** kernel is [-b -a 1 -a -b] / (1-2a-2b) */
    private List<Float> viewSharpeningCoefficientsToKernel(float a, float b) {
        float denominator = 1 - 2 * a - 2 * b;

        List<Float> kernel = new ArrayList<>(5);
        kernel.add(-b / denominator);
        kernel.add(-a / denominator);
        kernel.add(1 / denominator);
        kernel.add(-a / denominator);
        kernel.add(-b / denominator);

        return kernel;
    }

    @Override
    public String toString() {
        return "DisplayConfig{\n"
                + "  mDotPitchInMm="
                + mDotPitchInMm
                + ",\n"
                + "  mPanelResolution="
                + mPanelResolution
                + ",\n"
                + "  mNumViews="
                + mNumViews
                + ",\n"
                + "  mAlignmentOffset="
                + mAlignmentOffset
                + ",\n"
                + "  mViewSharpeningCoefficients="
                + mViewSharpeningCoefficients
                + ",\n"
                + "  mSystemDisparity="
                + mSystemDisparity
                + ",\n"
                + "}";
    }
}
