/*
 * Decompiled with CFR 0.152.
 */
package com.ledger.lib.transport;

import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.os.Build;
import android.util.Log;
import com.ledger.lib.LedgerException;
import com.ledger.lib.transport.GattCallback;
import com.ledger.lib.transport.LedgerDevice;
import com.ledger.lib.transport.LedgerWrapper;
import com.ledger.lib.utils.Dump;
import java.io.ByteArrayOutputStream;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

public class LedgerDeviceBLE
implements LedgerDevice {
    public static final UUID SERVICE_UUID = UUID.fromString("13D63400-2C97-0004-0000-4C6564676572");
    private static final UUID WRITE_CHARACTERISTIC_UUID = UUID.fromString("13D63400-2C97-0004-0002-4C6564676572");
    private static final UUID NOTIFY_CHARACTERISTIC_UUID = UUID.fromString("13D63400-2C97-0004-0001-4C6564676572");
    private static final UUID CLIENT_CHARACTERISTIC_CONFIG = UUID.fromString("00002902-0000-1000-8000-00805f9b34fb");
    private static final byte[] QUERY_MTU = new byte[]{8, 0, 0, 0, 0};
    private static final int DEFAULT_MTU = 20;
    private static final int DEFAULT_MAX_MTU = 100;
    private static final int DEFAULT_QUEUE_CAPACITY = 100;
    private static final int DEFAULT_TIMEOUT_MS = 15000;
    private static final String LOG_STRING = "LedgerDeviceBLE";
    private BluetoothGatt connection;
    private int timeout;
    private int maxMtu;
    private BluetoothGattCharacteristic characteristicWrite;
    private BluetoothGattCharacteristic characteristicNotify;
    private GattCallback gattCallback;
    private LinkedBlockingQueue<GattCallback.GattEvent> blockingQueue;
    private byte[] transferBuffer;
    private boolean opened;
    private boolean disconnected;
    private boolean debug;
    private int mtu;

    public LedgerDeviceBLE(BluetoothGatt connection, int timeoutMS, int maxMTU) {
        this.connection = connection;
        this.blockingQueue = new LinkedBlockingQueue();
        this.gattCallback = new GattCallback(this.blockingQueue);
        this.timeout = timeoutMS;
        this.maxMtu = maxMTU;
        this.setMtu(20);
    }

    public LedgerDeviceBLE(BluetoothGatt connection) {
        this(connection, 15000, 100);
    }

    public GattCallback getGattCallback() {
        return this.gattCallback;
    }

    private void setMtu(int mtu) {
        this.mtu = mtu;
        this.transferBuffer = new byte[mtu];
    }

    private void clearQueue() {
        GattCallback.GattEvent event;
        if (this.debug) {
            Log.d((String)LOG_STRING, (String)"Begin clear queue");
        }
        while ((event = this.blockingQueue.poll()) != null) {
            if (!this.debug) continue;
            Log.d((String)LOG_STRING, (String)("Dropping " + event.toString()));
        }
        if (this.debug) {
            Log.d((String)LOG_STRING, (String)"End clear queue");
        }
    }

    private GattCallback.GattEvent waitEvent(GattCallback.GattEventType eventType) throws LedgerException {
        return this.waitEvent(eventType, null, null, null);
    }

    private GattCallback.GattEvent waitEvent(GattCallback.GattEventType eventType, UUID uuid) throws LedgerException {
        return this.waitEvent(eventType, uuid, null, null);
    }

    private GattCallback.GattEvent waitEvent(GattCallback.GattEventType eventType, UUID uuid, GattCallback.GattEventType eventType2, UUID uuid2) throws LedgerException {
        GattCallback.GattEvent event;
        do {
            event = null;
            try {
                event = this.blockingQueue.poll(this.timeout, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException ex) {
                throw new LedgerException(LedgerException.ExceptionReason.INTERNAL_ERROR, (Throwable)ex);
            }
            if (event == null) {
                throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Timeout");
            }
            if (this.debug) {
                Log.d((String)LOG_STRING, (String)("Received " + event.toString()));
            }
            if (event.getStatus() != 0) {
                throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Unsuccessful event " + event.toString());
            }
            switch (event.getEventType()) {
                case GATT_MTU_CHANGED: {
                    this.setMtu(event.getMtu() - 3);
                    break;
                }
                case GATT_CONNECTION_STATE_CHANGE: {
                    switch (event.getNewState()) {
                        case 0: 
                        case 3: {
                            this.disconnected = true;
                            this.opened = false;
                            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Disconnected");
                        }
                    }
                }
            }
            if (!event.getEventType().equals((Object)eventType) || uuid != null && !event.getUuid().equals(uuid)) continue;
            return event;
        } while (eventType2 == null || !event.getEventType().equals((Object)eventType2) || uuid2 != null && !event.getUuid().equals(uuid2));
        return event;
    }

    @Override
    public void open() throws LedgerException {
        if (this.opened) {
            Log.d((String)LOG_STRING, (String)"Already opened");
            return;
        }
        if (this.disconnected) {
            Log.d((String)LOG_STRING, (String)"Disconnected");
            return;
        }
        this.clearQueue();
        if (!this.connection.discoverServices()) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to initiate GATT service discovery");
        }
        GattCallback.GattEvent event = this.waitEvent(GattCallback.GattEventType.GATT_SERVICES_DISCOVERED);
        this.characteristicWrite = null;
        this.characteristicNotify = null;
        List services = this.connection.getServices();
        for (BluetoothGattService service : services) {
            if (!service.getUuid().equals(SERVICE_UUID)) continue;
            List characteristics = service.getCharacteristics();
            for (BluetoothGattCharacteristic characteristic : characteristics) {
                if (characteristic.getUuid().equals(WRITE_CHARACTERISTIC_UUID)) {
                    this.characteristicWrite = characteristic;
                    continue;
                }
                if (!characteristic.getUuid().equals(NOTIFY_CHARACTERISTIC_UUID)) continue;
                this.characteristicNotify = characteristic;
            }
        }
        if (this.characteristicWrite == null || this.characteristicNotify == null) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to find all service characteristics");
        }
        if (!this.connection.setCharacteristicNotification(this.characteristicNotify, true)) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to enable local notifications");
        }
        BluetoothGattDescriptor descriptor = this.characteristicNotify.getDescriptor(CLIENT_CHARACTERISTIC_CONFIG);
        descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
        if (!this.connection.writeDescriptor(descriptor)) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to enable remote notifications");
        }
        this.waitEvent(GattCallback.GattEventType.GATT_DESCRIPTOR_WRITE, CLIENT_CHARACTERISTIC_CONFIG);
        if (Build.VERSION.SDK_INT >= 21) {
            if (!this.connection.requestMtu(this.maxMtu)) {
                throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to request MTU");
            }
            this.waitEvent(GattCallback.GattEventType.GATT_MTU_CHANGED);
        }
        this.characteristicWrite.setValue(QUERY_MTU);
        if (!this.connection.writeCharacteristic(this.characteristicWrite)) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to write query_mtu message");
        }
        this.waitEvent(GattCallback.GattEventType.GATT_CHARACTERISTIC_WRITE, WRITE_CHARACTERISTIC_UUID);
        event = this.waitEvent(GattCallback.GattEventType.GATT_CHARACTERISTIC_CHANGED, NOTIFY_CHARACTERISTIC_UUID);
        byte[] data = event.getData();
        Log.d((String)LOG_STRING, (String)("Device MTU answer " + Dump.dump(data)));
        this.opened = true;
    }

    @Override
    public byte[] exchange(byte[] apdu) throws LedgerException {
        GattCallback.GattEvent event = null;
        ByteArrayOutputStream response = new ByteArrayOutputStream();
        byte[] responseData = null;
        int offset = 0;
        if (!this.opened) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Device is not opened");
        }
        if (this.disconnected) {
            throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Device is disconnected");
        }
        this.clearQueue();
        if (this.debug) {
            Log.d((String)LOG_STRING, (String)("=> " + Dump.dump(apdu)));
        }
        apdu = LedgerWrapper.wrapCommandAPDU(apdu, this.mtu);
        while (offset != apdu.length) {
            int blockSize = apdu.length - offset > this.mtu ? this.mtu : apdu.length - offset;
            System.arraycopy(apdu, offset, this.transferBuffer, 0, blockSize);
            if (this.debug) {
                Log.d((String)LOG_STRING, (String)("=> Fragment " + Dump.dump(this.transferBuffer)));
            }
            this.characteristicWrite.setValue(this.transferBuffer);
            if (!this.connection.writeCharacteristic(this.characteristicWrite)) {
                throw new LedgerException(LedgerException.ExceptionReason.IO_ERROR, "Failed to write fragment");
            }
            event = this.waitEvent(GattCallback.GattEventType.GATT_CHARACTERISTIC_WRITE, WRITE_CHARACTERISTIC_UUID, GattCallback.GattEventType.GATT_CHARACTERISTIC_CHANGED, NOTIFY_CHARACTERISTIC_UUID);
            offset += blockSize;
        }
        if (!event.getEventType().equals((Object)GattCallback.GattEventType.GATT_CHARACTERISTIC_CHANGED)) {
            event = null;
        }
        while (responseData == null) {
            if (event == null) {
                event = this.waitEvent(GattCallback.GattEventType.GATT_CHARACTERISTIC_CHANGED, NOTIFY_CHARACTERISTIC_UUID);
            }
            byte[] data = event.getData();
            if (this.debug) {
                Log.d((String)LOG_STRING, (String)("<= Fragment " + Dump.dump(data)));
            }
            response.write(data, 0, data.length);
            responseData = LedgerWrapper.unwrapResponseAPDU(response.toByteArray(), this.mtu);
            event = null;
        }
        if (this.debug) {
            Log.d((String)LOG_STRING, (String)("<= " + Dump.dump(responseData)));
        }
        return responseData;
    }

    @Override
    public void close() throws LedgerException {
    }

    @Override
    public void setDebug(boolean debugFlag) {
        this.debug = debugFlag;
    }

    @Override
    public boolean isOpened() {
        return this.opened;
    }
}

