package com.ksc.mission.base.s3;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.SdkClientException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.event.ProgressEvent;
import com.amazonaws.event.ProgressEventType;
import com.amazonaws.event.ProgressListener;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.iterable.S3Objects;
import com.amazonaws.services.s3.model.BucketVersioningConfiguration;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest.KeyVersion;
import com.amazonaws.services.s3.model.DeleteVersionRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.SetBucketVersioningConfigurationRequest;
import com.amazonaws.services.s3.model.VersionListing;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import com.amazonaws.util.StringUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ksc.mission.base.interfaces.HASHTAGS;
import com.ksc.mission.base.interfaces.IS3InputStream;
import com.ksc.mission.base.relatedobjects.OwnedService;
import com.ksc.mission.base.util.StringUtil;

public class S3ClientConnector extends OwnedService {
	private static final long serialVersionUID = 1L;
	private static final int DEFAULT_BUFFER_SIZE = 512 * 1024;
	transient final protected AmazonS3 s3Client;
	final protected String s3Type;   // DELL or AWS
	final protected String bucketName; // must be lowercase
	final protected boolean versioning;
	protected TransferManager transferManager;
	
	public static AmazonS3 defaultS3Client(S3Properties properties) {
		if(properties.type.equalsIgnoreCase("DELL"))
			return dellS3Client(properties);
		else
			return amazonS3Client(properties);
	}

	
	public static AmazonS3 amazonS3Client(S3Properties properties) {
		return amazonS3ClientFor(properties.url, properties.region, new CredentialsProvider(properties));
	}
	
	public static AmazonS3 dellS3Client(S3Properties properties) {
		return dellS3ClientFor(properties.url, new CredentialsProvider(properties));
	}
	
	public static AmazonS3 amazonS3ClientFor(String endpoint, String region, AWSCredentials credentialsProvider) {
	    return AmazonS3ClientBuilder.standard()
	                            .withCredentials(new AWSStaticCredentialsProvider(credentialsProvider))
	                            .withPathStyleAccessEnabled(true)
	                            .withEndpointConfiguration(new EndpointConfiguration(endpoint, region))
	                            .build();
    }

	public static AmazonS3 dellS3ClientFor(String endpoint, AWSCredentials credentialsProvider) {
//		TODO: Change to use the Dell EMC ECS S3Config class
	    return AmazonS3ClientBuilder.standard()
                .withCredentials(new AWSStaticCredentialsProvider(credentialsProvider))
                .withPathStyleAccessEnabled(true)
                .withEndpointConfiguration(new EndpointConfiguration(endpoint, ""))
                .build();
    }


	public S3ClientConnector(S3Properties properties) {
		this(defaultS3Client(properties), properties.type, properties.bucketName, properties.versioning);
	}
	
	public static S3ClientConnector from(String environment) {
		return new S3ClientConnector(S3Properties.from(environment));
	}
	
	public static S3ClientConnector forBucket(String environment, String bucketName, boolean versioning) {
		return forBucket(S3Properties.from(environment), bucketName, versioning);
	}
	
	public static S3ClientConnector forBucket(S3Properties properties, String bucketName, boolean versioning) {
		return new S3ClientConnector(defaultS3Client(properties), properties.type, bucketName, versioning);
	}
	
	public S3ClientConnector(AmazonS3 s3Client, String s3Type, String bucketName, boolean versioning) {
		this.s3Client = s3Client;
		this.versioning = versioning;
		this.s3Type = s3Type; 
		this.bucketName = bucketName; // must be lowercase for AWS
		if(!doesBucketExist()) {
			createBucket();
			setBucketVersioning(versioning);
		}
	}
		
	@Override
	public void start() {	
		subscribe(HASHTAGS.CREATED, (msg)-> {
				try {
					createOrReplaceObject("" + msg.getPayload(), new ObjectMapper().writer().writeValueAsString(msg.getPayload()));
				} catch (JsonProcessingException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				logDebug("createOrReplaceObject - " + msg);
		});
		subscribe(HASHTAGS.S3_LIST, (msg)-> {
			String prefix = (String)msg.getPayload();
			Stream<S3ObjectSummary> stream = null;
			if(prefix == null || prefix.isEmpty())
				stream = stream();
			else
				stream = streamWithPrefix(prefix);
			logInfo("File List for Bucket: " + bucketName);
			stream.forEach(eaSummary ->
				logInfo("    " + eaSummary.getKey() + " " + eaSummary.getSize() + " " + eaSummary.getLastModified()));
		});
		subscribe(HASHTAGS.S3_LIST_VERSIONS, (msg)-> {
			String prefix = (String)msg.getPayload();
			if(prefix == null)
				prefix = "";
			streamVersionsWithPattern(prefix).forEach(eaSummary ->
				logDebug(eaSummary.getKey() + " " + eaSummary.getLastModified()));
		});
		
		subscribe(HASHTAGS.S3_DELETE_ALL, (msg)-> {
			String bName = (String)msg.getPayload();
			if(bucketName.equals(bName))
				deleteAllObjects();
		});
		
		subscribe(HASHTAGS.S3_DELETE, (msg)-> {
			String[] args = (String[])msg.getPayload();
			if(getId().equals(args[0]))
				deleteAllWithPattern(args[1], args[2]);
		});
    }
	
	public void setBucketVersioning(boolean versioning) {
		if(versioning) {
			BucketVersioningConfiguration configuration = 
					new BucketVersioningConfiguration().withStatus(versioning ? "Enabled" : "Disabled");
			SetBucketVersioningConfigurationRequest setBucketVersioningConfigurationRequest = 
					new SetBucketVersioningConfigurationRequest(bucketName,configuration);
			s3Client.setBucketVersioningConfiguration(setBucketVersioningConfigurationRequest);
		}
	}

	public Date getUtcTime() {
		String timeNowFile = "TIME_NOW-" + UUID.randomUUID();
		createOrReplaceObject(timeNowFile, "");
		S3Object file = getObject(timeNowFile);
		Date lastMod = file.getObjectMetadata().getLastModified();
		deleteObject(timeNowFile);
		return lastMod;
	}
	
	public void deleteAllObjects() {
		 deleteAllObjects(() -> stream()); 
	}
	
	public void deleteAllWithPrefix(String prefix, String excludePattern) {
		deleteAllObjects(() -> {
			Stream<S3ObjectSummary> stream = streamWithPrefix(prefix);
			if(excludePattern != null && !excludePattern.trim().isEmpty())
				stream = stream.filter(ea -> !StringUtil.matches(ea.getKey(), excludePattern));
			 return stream;
		});
	}
	
	public void deleteAllWithPattern(String includePattern, String excludePattern) {
		deleteAllObjects(() -> {
		Stream<S3ObjectSummary> stream = streamWithPattern(includePattern);
		if(excludePattern != null && !excludePattern.trim().isEmpty())
			stream = stream.filter(ea -> !StringUtil.matches(ea.getKey(), excludePattern));
		 return stream;
	});
	}
	
	// will try not to delete objects added after this call begins
	public void deleteAllObjects(Supplier<Stream<S3ObjectSummary>> streamSupplier) {
		int batchSize = 1000;
		Date now = getUtcTime();
		while(true) {
			long startTime = System.currentTimeMillis();
			List<String> keys = streamSupplier.get()
					.filter(ea -> ea.getLastModified().before(now))
					.map(ea -> ea.getKey())
					.limit(batchSize)
					.collect(Collectors.toList());
			if(keys.isEmpty())
				return;
			deleteAllObjects(keys);
			long endTime = System.currentTimeMillis();
			logInfo("Deleted - " + keys.size() + " Time - " + (endTime - startTime));
		}
	}
	protected Stream<String> urlEncoded(Stream<String> stream) {
		return stream
				.map(key -> {
					String encodedKey;
					try {
						encodedKey = URLEncoder.encode(key, StandardCharsets.UTF_8.toString());
					} catch (UnsupportedEncodingException e) {
						e.printStackTrace();
						return key;
					}
					return (String)encodedKey;
				});
	}
	
	public void deleteAllObjects(List<String> keys) {
		try {
			List<KeyVersion> keyVersions = keys.stream()
					.map(ea -> new KeyVersion(ea))
					.collect(Collectors.toList());
			DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucketName).withKeys(keyVersions);
			s3Client.deleteObjects(deleteRequest);
		} catch(Exception ex) {
			keys.forEach(key -> deleteObject(key));
		}
	}
	
	public void deleteObject(String key) {
		DeleteObjectRequest deleteRequest = new DeleteObjectRequest(bucketName, key);
		s3Client.deleteObject(deleteRequest);
	}
	
	public void deleteObjectVersion(String key, String version) {
		DeleteVersionRequest deleteVersionRequest = new DeleteVersionRequest(bucketName, key, version);
		s3Client.deleteVersion(deleteVersionRequest);
	}
	
	public TransferManager getTransferManager() {
		if(transferManager == null)
			transferManager = TransferManagerBuilder.standard()
									                .withS3Client(s3Client)
									                .build();
		return transferManager;
	}
	
    public void putObject(String key, byte[] contentBytes) {
        InputStream is = new ByteArrayInputStream(contentBytes);
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentType("application/octet-stream");
        metadata.setContentLength(contentBytes.length);
        s3Client.putObject(new PutObjectRequest(bucketName, key, is, metadata));
    }
    
	public void createOrReplaceObject(String key, String contents) {
		s3Client.putObject(bucketName, key, contents);
	}
	
	public void createOrReplaceObjectAsync(String key, String contents) {
		try {
			createOrReplaceObjectAsync(key, contents.getBytes("UTF8"));
		} catch (UnsupportedEncodingException e) {}
	}
	
	public void createOrReplaceObject(String key, byte[] bytes) {
		createOrReplaceObject(key, new ByteArrayInputStream(bytes), bytes.length, true);
		if(!exists(key))
			putObject(key, bytes);
	}
	
	public void createOrReplaceObjectAsync(String key, byte[] bytes) {
		createOrReplaceObject(key, new ByteArrayInputStream(bytes), bytes.length, false);
	}
	
	public void createOrReplaceObject(String key, IS3InputStream s3InputStream) {
		createOrReplaceObject(key, s3InputStream, s3InputStream.getContentLength(), true);
	}
	
	public void createOrReplaceObject(String key, IS3InputStream s3InputStream, boolean wait) {
		createOrReplaceObject(key, s3InputStream, s3InputStream.getContentLength(), wait);
	}
	
	public void createOrReplaceObject(String key, InputStream inputStream, long size, boolean wait) {
		try {
			ObjectMetadata metadata = new ObjectMetadata();
			metadata.setContentLength(size);
			Upload upload = getTransferManager().upload(bucketName, key, inputStream, metadata);
			if (wait)
				upload.waitForCompletion();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public void createOrReplaceObjectAsync(String key, Supplier<IS3InputStream> s3InputStreamSupplier, Consumer<String> onCompletion, Consumer<Exception> onError) {
		createOrReplaceObjectAsync(key, s3InputStreamSupplier, s3InputStreamSupplier.get().getContentLength(), onCompletion,  onError);
	}
	
	public void createOrReplaceObjectAsync(String key, Supplier<? extends InputStream> inputStreamSupplier, long size, 
			Consumer<String> onCompletion, Consumer<Exception> onError) {
		try {
			ObjectMetadata metadata = new ObjectMetadata();
			metadata.setContentLength(size);
			Upload upload = getTransferManager().upload(bucketName, key, inputStreamSupplier.get(), metadata);
			upload.addProgressListener(new ProgressListener() {
			    public void progressChanged(ProgressEvent progressEvent) {
			        if (progressEvent.getEventType() == ProgressEventType.TRANSFER_COMPLETED_EVENT) {
							if(!exists(key))
								try {
									createOrReplaceObject(key, inputStreamSupplier.get().readAllBytes());
								} catch (IOException e) {
			        				onError.accept(new Exception("S3 file failed to write - " + key, e));
			        				return;
								}
			        			if(!exists(key)) {
			        				onError.accept(new Exception("S3 file failed to write - " + key));
			        				return;
			        			}
							else
								onCompletion.accept(key);
			        }
			        else if (progressEvent.getEventType() == ProgressEventType.TRANSFER_FAILED_EVENT)
			            onError.accept(new Exception("Transfer failed - " + key));  
			    }
			});
		} catch (Exception e) {
			onError.accept(new Exception("Transfer failed - " + key));
		}
	}
	
	public S3Object getObject(String key) {
		return s3Client.getObject(new GetObjectRequest(bucketName, key));
	}
	
	public S3Object getObject(String key, String versionId) {
		return s3Client.getObject(new GetObjectRequest(bucketName, key, versionId));
	}
	
	public void createBucket() {
		if(!doesBucketExist())
			s3Client.createBucket(bucketName);
	}
	
	public void deleteBucket() {
		s3Client.deleteBucket(bucketName);
	}
	
	public void createOrReplaceObject(String key, File file) {
		s3Client.putObject(bucketName, key, file);
	}
	
	public boolean doesBucketExist() {
		return s3Client.doesBucketExistV2(bucketName);
	}

	public boolean exists(String key) {
		return s3Client.doesObjectExist(bucketName, key);
	}

	public Stream<S3ObjectSummary> streamWithPrefix(String prefix) {
		S3Objects s3Objects = S3Objects.withPrefix(s3Client, bucketName, prefix);
		return StreamSupport.stream(s3Objects.spliterator(), false);
	}
	
	public Stream<S3ObjectSummary> streamWithPattern(String pattern) {
		Stream<S3ObjectSummary> stream = null;
		if(pattern.startsWith("*")) 
			stream = stream();
		else {
			String prefix = pattern.split("\\*")[0];
			stream = streamWithPrefix(prefix);
		}
		return stream
				.filter(ea -> StringUtil.matches(ea.getKey(), pattern));
	}
	
	public Stream<S3ObjectSummary> stream() {
		return StreamSupport.stream(S3Objects.inBucket(s3Client, bucketName).spliterator(), false);
	}
	
	public List<S3VersionSummary> listVersions(String prefix) {
        ListVersionsRequest request = new ListVersionsRequest()
                .withBucketName(bucketName)
                .withPrefix(prefix)
                .withMaxResults(100);
        VersionListing result = s3Client.listVersions(request); 
		return result.getVersionSummaries();
	}

	public List<S3VersionSummary> streamVersionsWithPattern(String pattern) {
		List<String> names = streamWithPattern(pattern)
				.map(ea -> ea.getKey())
				.collect(Collectors.toList());
		return names.stream().flatMap(ea -> listVersions(ea).stream()).collect(Collectors.toList());		
	}
	
	public InputStreamReader openInputStreamReader(String key) {
		return new InputStreamReader(openPositionableReadStream(key));
	}

	public InputStreamReader openInputStreamReader(String key, String versionId) {
		return new InputStreamReader(openPositionableReadStream(key, versionId));
	}

	public PositionableReadStream openPositionableReadStream(String key) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, ObjectMetadata metadata, long position) {
		return new PositionableReadStream(this, key, metadata, position, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, long startPosition, long endPosition) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), startPosition, DEFAULT_BUFFER_SIZE, endPosition);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId, long startPosition, long endPosition) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), startPosition, DEFAULT_BUFFER_SIZE, endPosition);
	}

	public PositionableReadStream openPositionableReadStream(String key, ObjectMetadata metadata, long startPosition, long endPosition) {
		return new PositionableReadStream(this, key, metadata, startPosition, DEFAULT_BUFFER_SIZE, endPosition);
	}

	
	public ObjectMetadata getObjectMetadata(String key, String versionId) {
		return s3Client.getObjectMetadata(new GetObjectMetadataRequest(bucketName, key, versionId));
	}
	
	public boolean getRange(String file, long position, int[] buffer) {
		GetObjectRequest request = new GetObjectRequest(bucketName, file).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(String file, String versionId, long position, int[] buffer) {		
		GetObjectRequest request = new GetObjectRequest(bucketName, file, versionId).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(GetObjectRequest request, int[] buffer) {		
		try(S3Object s3Object = s3Client.getObject(request)) {
			int index = 0;
			S3ObjectInputStream stream = s3Object.getObjectContent();
			int next;
			while((next = stream.read()) != -1)
				buffer[index++] = next;
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	public String getVersionId(String key) {
		// TODO Auto-generated method stub
		return null;
	}


}