package com.ksc.mission.base.s3;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.Scanner;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import java.util.zip.ZipInputStream;

import com.amazonaws.services.s3.model.ObjectMetadata;

public class PositionableReadStream extends InputStream {
	protected int maxBufferSize;
	protected final S3ClientConnector clientConnector;
	protected Scanner scanner; 
	protected int[] buffer; 
	protected long bufferStartPosition; 
	protected long bufferEndPosition; 
	protected final String key;
	protected final ObjectMetadata metadata;
	protected long position = 0;
	protected long objectLength;

    public PositionableReadStream(S3ClientConnector clientConnector, String key, ObjectMetadata metadata, long position, int maxBufferSize) {
        this.clientConnector = clientConnector;
        this.key = key;
        this.metadata = metadata;
        this.objectLength = getLength();
        this.position = position;
        this.maxBufferSize = maxBufferSize; 
        this.buffer = new int[maxBufferSize];
        getNextPage(position);
    }

    public PositionableReadStream(S3ClientConnector clientConnector, String key, ObjectMetadata metadata, long position, int maxBufferSize, long readLimit) {
        this.clientConnector = clientConnector;
        this.key = key;
        this.metadata = metadata;
        this.objectLength = Math.min(getLength(), readLimit);
        this.position = position;
        long dataSize = objectLength - position;
        this.maxBufferSize = (int)Math.min(dataSize, (long)maxBufferSize); 
        this.buffer = new int[maxBufferSize];
        getNextPage(position);
    }

    public ZipInputStream asZipInputStream() {
    	return new ZipInputStream(this);
    }
    
    @Override
    public int read() {
        if (position >= objectLength) 
			return -1;
        if((position > bufferEndPosition) 
		|| (position < bufferStartPosition)) {
        	if(!getNextPage(position))
        		return -1;
        }
        int b = buffer[(int)(position - bufferStartPosition)];
        position++;
        return b;
    }

    public boolean getNextPage(long position) { 
    	bufferStartPosition = position;
    	bufferEndPosition = bufferStartPosition + buffer.length - 1;
    	return clientConnector.getRange(key, metadata.getVersionId(), position, buffer);
    }
    
    public byte[] toByteArray() { 
    	long currentPosition = position;
    	position = 0;
    	ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    	byte next;
    	while((next = (byte)read()) != -1)
    		byteStream.write(next);
    	position = currentPosition;
    	return byteStream.toByteArray();
    }
    
    public PositionableReadStream clone() {
    	return clientConnector.openPositionableReadStream(key, metadata, position);
    }
    
	public long indexOf(String string) {
		return indexOf(string.getBytes());
	}
	
    public long indexOf(byte[] bytes) {
    	long currentPosition = position;
    	while(!nextMatches(bytes)) 
    		if(read() == -1) {
    	    	position = currentPosition;
    			return -1;
    		}
    	long ans = position;
    	position = currentPosition;
    	return ans;
    }
    
	public boolean nextMatches(String string) {
		return nextMatches(string.getBytes());
	}
	
    public boolean nextMatches(byte[] bytes) {
    	int bytesLastIndex = bytes.length - 1;
    	long endReadPosition = position + bytesLastIndex;
    	if((endReadPosition > bufferEndPosition)
    			&& !getNextPage(position))
    		return false;   		
    	long currentPosition = position;
    	for(int i = 0; i <= bytesLastIndex; i++)
    		if(bytes[i] != read()) {
    	    	position = currentPosition;
    			return false;
    		}
    	position = currentPosition;
    	return true;
    }
    
    public byte[] peekFor(long length) {
    	if((position + length > bufferEndPosition)
    			&& !getNextPage(position))
    		return null;   		
    	long currentPosition = position;
    	byte[] ans = readTo(position + length);
    	position = currentPosition;
    	return ans;
    }
    
    public byte[] read(int length) {
    	return readTo(position + length);
    }
    
    public byte[] readTo(long endPosition) {
       	ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
    	long end = Math.min(endPosition, objectLength - 1);
    	int length = (int)(end - position);
    	if(length < 0)
    		return null;
    	for(int i = 0; i < length; i++)
    		byteStream.write(read());
    	return byteStream.toByteArray();
    }
    
	public InputStreamReader asInputStreamReader() {
		return new InputStreamReader(this);
	}

	public InputStreamReader asInputStreamReader(String codepage) {
		try {
			return new InputStreamReader(this, codepage);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

    
    public IntStream intStream() { 
    	return IntStream.generate(() -> read()).limit(objectLength - position);      
    } 
    
    public Stream<Character> characterStream() {
    	InputStreamReader reader = new InputStreamReader(this);
    	return (Stream<Character>) IntStream.generate(() -> {
							try {
								return reader.read();
							} catch (IOException e) {
								e.printStackTrace();
								return -1;
							}
				 		})
    			.takeWhile(each -> each != -1)
		    	.mapToObj(i -> (char)i);     
	   }
    
    public Scanner scanner() {
    	if(scanner == null)
    		scanner = new Scanner(this); 
    	return scanner;
    }
    
    public Stream<String> stringStream() {
    	final Scanner scanner = scanner();
    	return Stream.generate(() -> scanner.next());     
    }

	public long getPosition() {
		return position;
	}

	public void setPosition(long position) {
		this.position = position;
	}

	public int occurrences(String string) {
		int count = 0;
		byte[] bytes = string.getBytes();
		return occurrences(bytes);
	}
	public int occurrences(byte[] bytes) {
		int count = 0;
    	long currentPosition = position;
		if(nextMatches(bytes))
			count++;
		while(read() != -1)
			if(nextMatches(bytes))
				count++;
		position =currentPosition;
		return count;
	}

	public String getVersionId() {
		return metadata.getVersionId();
	}
	
	public long getLength() {
		return metadata.getContentLength();
	}    
    
}