package com.ksc.mission.base.s3;

import java.io.File;
import java.util.List;
import java.util.stream.Collectors;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.BucketVersioningConfiguration;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest.KeyVersion;
import com.amazonaws.services.s3.model.DeleteVersionRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.SetBucketVersioningConfigurationRequest;
import com.amazonaws.services.s3.model.VersionListing;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ksc.mission.base.interfaces.HASHTAGS;
import com.ksc.mission.base.relatedobjects.OwnedService;
import com.ksc.mission.base.util.StringUtil;

public class S3ClientConnector extends OwnedService {
	private static final long serialVersionUID = 1L;
	private static final int DEFAULT_BUFFER_SIZE = 256 * 256;
	transient final protected AmazonS3 s3Client;
	final protected String bucketName; // must be lowercase
//	final protected boolean versioning;
	
	public static AmazonS3 defaultS3Client() {
		return s3ClientFor("http://192.168.99.100:32768/","us-west-2", new CredentialsProvider());
	}
	
	public static AmazonS3 s3ClientFor(String endpoint, String region, AWSCredentials credentialsProvider) {
	    return AmazonS3ClientBuilder.standard()
	                            .withCredentials(new AWSStaticCredentialsProvider(credentialsProvider))
	                            .withPathStyleAccessEnabled(true)
	                            .withEndpointConfiguration(new EndpointConfiguration(endpoint, region))
	                            .build();
    }


	public static S3ClientConnector forBucket(String bucketName, boolean versioning) {
		return new S3ClientConnector(defaultS3Client(), "mybucket", versioning);
	}
	
	public S3ClientConnector(AmazonS3 s3Client, String bucketName, boolean versioning) {
		this.s3Client = s3Client;
		this.bucketName = bucketName; // must be lowercase
		setBucketVersioning(versioning);
	}
		
	@Override
	public void start() {	
		subscribe(HASHTAGS.CREATED, (msg)-> {
				try {
					createOrReplaceObject("" + msg.getPayload(), new ObjectMapper().writer().writeValueAsString(msg.getPayload()));
				} catch (JsonProcessingException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				logDebug("createOrReplaceObject - " + msg);
		});
		subscribe(HASHTAGS.S3_LIST, (msg)-> {
			String prefix = (String)msg.getPayload();
			List<S3ObjectSummary> list = null;
			if(prefix == null || prefix.isEmpty())
				list = listWithPrefix(prefix);
			else
				list = list();
			for(S3ObjectSummary eaSummary : list)
				logInfo("" + eaSummary);
		});
		subscribe(HASHTAGS.S3_LIST_VERSIONS, (msg)-> {
			String prefix = (String)msg.getPayload();
			List<S3VersionSummary> list = null;
			if(prefix == null)
				prefix = "";
			list = listVersionsWithPattern(prefix);
			for(S3VersionSummary eaSummary : list)
				logDebug(eaSummary.getKey() + " " + eaSummary.getLastModified());
		});
		subscribe(HASHTAGS.S3_DELETE_ALL, (msg)-> {
			String bName = (String)msg.getPayload();
			if(bucketName.equals(bName))
				deleteAllObjects();
		});
    }
	
	public void setBucketVersioning(boolean versioning) {
		BucketVersioningConfiguration configuration = 
				new BucketVersioningConfiguration().withStatus(versioning ? "Enabled" : "Disabled");
		SetBucketVersioningConfigurationRequest setBucketVersioningConfigurationRequest = 
				new SetBucketVersioningConfigurationRequest(bucketName,configuration);
		s3Client.setBucketVersioningConfiguration(setBucketVersioningConfigurationRequest);
	}

	public void deleteAllObjects() {
		List<KeyVersion> keyVersions = list().stream()
				.map(ea -> new KeyVersion(ea.getKey()))
				.collect(Collectors.toList());
		deleteAllObjects(keyVersions);
	}
	
	public void deleteAllObjects(List<KeyVersion> keyVersions) {
		DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucketName).withKeys(keyVersions);
		s3Client.deleteObjects(deleteRequest);
	}
	
	public void deleteObject(String key) {
		DeleteObjectRequest deleteRequest = new DeleteObjectRequest(bucketName, key);
		s3Client.deleteObject(deleteRequest);
	}
	
	public void deleteObjectVersion(String key, String version) {
		DeleteVersionRequest deleteVersionRequest = new DeleteVersionRequest(bucketName, key, version);
		s3Client.deleteVersion(deleteVersionRequest);
	}
	
	public void createOrReplaceObject(String key, String contents) {
		s3Client.putObject(bucketName, key, contents);
	}
	
	public void getObject(String key) {
		s3Client.getObject(new GetObjectRequest(bucketName, key));
	}
	
	public void getObject(String key, String versionId) {
		s3Client.getObject(new GetObjectRequest(bucketName, key, versionId));
	}
	
	public void createOrReplaceObject(String key, File file) {
		s3Client.putObject(bucketName, key, file);
	}
	
	public boolean doesBucketExist(String bucketName) {
		return s3Client.doesBucketExistV2(bucketName);
	}

	public boolean exists(String key) {
		return s3Client.doesObjectExist(bucketName, key);
	}

	public List<S3ObjectSummary> listWithPrefix(String prefix) {
		ObjectListing result = s3Client.listObjects(bucketName, prefix);
		return result.getObjectSummaries();
	}

	public List<S3ObjectSummary> listWithPattern(String pattern) {
		List<S3ObjectSummary> list = null;
		if(pattern.startsWith("*")) 
			list = list();
		else {
			String prefix = pattern.split("\\*")[0];
			list = listWithPrefix(prefix);
		}
		return list.stream()
				.filter(ea -> StringUtil.matches(ea.getKey(), pattern))
				.collect(Collectors.toList());
	}

	public List<S3ObjectSummary> list() {
		ObjectListing result = s3Client.listObjects(bucketName);
		return result.getObjectSummaries();
	}
	
	public List<S3VersionSummary> listVersions(String prefix) {
        ListVersionsRequest request = new ListVersionsRequest()
                .withBucketName(bucketName)
                .withPrefix(prefix)
                .withMaxResults(100);
        VersionListing result = s3Client.listVersions(request); 
		return result.getVersionSummaries();
	}

	public List<S3VersionSummary> listVersionsWithPattern(String pattern) {
		List<String> names = listWithPattern(pattern).stream()
				.map(ea -> ea.getKey())
				.collect(Collectors.toList());
		return names.stream().flatMap(ea -> listVersions(ea).stream()).collect(Collectors.toList());		
	}
	
	public PositionableReadStream openPositionableReadStream(String key) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, ObjectMetadata metadata, long position) {
		return new PositionableReadStream(this, key, metadata, position, DEFAULT_BUFFER_SIZE);
	}

	public ObjectMetadata getObjectMetadata(String key, String versionId) {
		return s3Client.getObjectMetadata(new GetObjectMetadataRequest(bucketName, key, versionId));
	}
	
	public boolean getRange(String file, long position, int[] buffer) {
		GetObjectRequest request = new GetObjectRequest(bucketName, file).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(String file, String versionId, long position, int[] buffer) {		
		GetObjectRequest request = new GetObjectRequest(bucketName, file, versionId).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(GetObjectRequest request, int[] buffer) {		
		try(S3Object s3Object = s3Client.getObject(request)) {
			int index = 0;
			S3ObjectInputStream stream = s3Object.getObjectContent();
			int next;
			while((next = stream.read()) != -1)
				buffer[index++] = next;
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	public String getVersionId(String key) {
		// TODO Auto-generated method stub
		return null;
	}


}