package com.mission.base.s3;

import java.io.File;
import java.util.List;
import java.util.stream.Collectors;

import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.BucketVersioningConfiguration;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.DeleteObjectsRequest.KeyVersion;
import com.amazonaws.services.s3.model.DeleteVersionRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.SetBucketVersioningConfigurationRequest;
import com.amazonaws.services.s3.model.VersionListing;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mission.base.interfaces.HASHTAGS;
import com.mission.base.relatedobjects.OwnedService;
import com.mission.base.util.StringUtil;

/**
 * Upload a file to an Amazon S3 bucket.
 *
 * This code expects that you have AWS credentials set up per:
 * http://docs.aws.amazon.com/java-sdk/latest/developer-guide/setup-credentials.html
 */
public class S3ClientConnector extends OwnedService {
	private static final long serialVersionUID = 1L;
	private static final int DEFAULT_BUFFER_SIZE = 256 * 256;
	protected AmazonS3 s3Client = null;
	final protected String endpoint;
	final protected String region;
	final protected String bucketName; // must be lowercase
	final protected boolean versioning;

	public S3ClientConnector() {
		super();
		endpoint = "http://192.168.99.100:32768/";
		region = "us-west-2";
		bucketName = "mybucket"; // must be lowercase
		versioning = true;
	}
	
	public S3ClientConnector(String bucketName, boolean versioning) {
		super();
		endpoint = "http://192.168.99.100:32768/";
		region = "us-west-2";
		this.bucketName = bucketName; // must be lowercase
		this.versioning = versioning;
	}
	
	public S3ClientConnector(String endpoint, String region, String bucketName, boolean versioning) {
		super();
		this.endpoint = endpoint;
		this.region = region;
		this.bucketName = bucketName.toLowerCase(); // must be lowercase
		this.versioning = versioning;
	}
	
	@Override
	public void init()
    {
        s3Client = AmazonS3ClientBuilder.standard()
                                .withCredentials(new AWSStaticCredentialsProvider(new CredentialsProvider()))
                                .withPathStyleAccessEnabled(true)
                                .withEndpointConfiguration(new EndpointConfiguration(endpoint, region))
                                .build();
		if(!doesBucketExist(bucketName)) 
			getS3Client().createBucket(bucketName);
        setBucketVersioning();
		subscribe(HASHTAGS.CREATED, (msg)-> {
				try {
					createOrReplaceObject("" + msg.getPayload(), new ObjectMapper().writer().writeValueAsString(msg.getPayload()));
				} catch (JsonProcessingException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				System.out.println("createOrReplaceObject - " + msg);
		});
		subscribe(HASHTAGS.S3_LIST, (msg)-> {
			String prefix = (String)msg.getPayload();
			List<S3ObjectSummary> list = null;
			if(prefix == null || prefix.isEmpty())
				list = listWithPrefix(prefix);
			else
				list = list();
			for(S3ObjectSummary eaSummary : list)
				System.out.println("" + eaSummary);
		});
		subscribe(HASHTAGS.S3_LIST_VERSIONS, (msg)-> {
			String prefix = (String)msg.getPayload();
			List<S3VersionSummary> list = null;
			if(prefix == null)
				prefix = "";
			list = listVersionsWithPattern(prefix);
			for(S3VersionSummary eaSummary : list)
				System.out.println(eaSummary.getKey() + " " + eaSummary.getLastModified());
		});
		subscribe(HASHTAGS.S3_DELETE_ALL, (msg)-> {
			String bName = (String)msg.getPayload();
			if(bucketName.equals(bName))
				deleteAllObjects();
		});
    }
	
	public void setBucketVersioning() {
		BucketVersioningConfiguration configuration = 
				new BucketVersioningConfiguration().withStatus(versioning ? "Enabled" : "Disabled");
		SetBucketVersioningConfigurationRequest setBucketVersioningConfigurationRequest = 
				new SetBucketVersioningConfigurationRequest(bucketName,configuration);
		s3Client.setBucketVersioningConfiguration(setBucketVersioningConfigurationRequest);
		BucketVersioningConfiguration conf = s3Client.getBucketVersioningConfiguration(bucketName);
		System.out.println("bucket versioning configuration status:    " + conf.getStatus());
	}

	public AmazonS3 getS3Client() {
		return s3Client; 
	}

	public void setS3Client(AmazonS3 s3Client) {
		this.s3Client = s3Client;
	}

	public void deleteAllObjects() {
		List<KeyVersion> keyVersions = list().stream()
				.map(ea -> new KeyVersion(ea.getKey()))
				.collect(Collectors.toList());
		deleteAllObjects(keyVersions);
	}
	
	public void deleteAllObjects(List<KeyVersion> keyVersions) {
		DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucketName).withKeys(keyVersions);
		s3Client.deleteObjects(deleteRequest);
	}
	
	public void deleteObject(String key) {
		DeleteObjectRequest deleteRequest = new DeleteObjectRequest(bucketName, key);
		s3Client.deleteObject(deleteRequest);
	}
	
	public void deleteObjectVersion(String key, String version) {
		DeleteVersionRequest deleteVersionRequest = new DeleteVersionRequest(bucketName, key, version);
		s3Client.deleteVersion(deleteVersionRequest);
	}
	
	public void createOrReplaceObject(String key, String contents) {
		getS3Client().putObject(bucketName, key, contents);
	}
	
	public void getObject(String key) {
		getS3Client().getObject(new GetObjectRequest(bucketName, key));
	}
	
	public void getObject(String key, String versionId) {
		getS3Client().getObject(new GetObjectRequest(bucketName, key, versionId));
	}
	
	public void createOrReplaceObject(String key, File file) {
		getS3Client().putObject(bucketName, key, file);
	}
	
	public boolean doesBucketExist(String bucketName) {
		return getS3Client().doesBucketExistV2(bucketName);
	}

	public boolean exists(String key) {
		return getS3Client().doesObjectExist(bucketName, key);
	}

	public List<S3ObjectSummary> listWithPrefix(String prefix) {
		ObjectListing result = getS3Client().listObjects(bucketName, prefix);
		return result.getObjectSummaries();
	}

	public List<S3ObjectSummary> listWithPattern(String pattern) {
		List<S3ObjectSummary> list = null;
		if(pattern.startsWith("*")) 
			list = list();
		else {
			String prefix = pattern.split("\\*")[0];
			list = listWithPrefix(prefix);
		}
		return list.stream()
				.filter(ea -> StringUtil.matches(ea.getKey(), pattern))
				.collect(Collectors.toList());
	}

	public List<S3ObjectSummary> list() {
		ObjectListing result = getS3Client().listObjects(bucketName);
		return result.getObjectSummaries();
	}
	
	public List<S3VersionSummary> listVersions(String prefix) {
        ListVersionsRequest request = new ListVersionsRequest()
                .withBucketName(bucketName)
                .withPrefix(prefix)
                .withMaxResults(100);
        VersionListing result = s3Client.listVersions(request); 
		return result.getVersionSummaries();
	}

	public List<S3VersionSummary> listVersionsWithPattern(String pattern) {
		List<String> names = listWithPattern(pattern).stream()
				.map(ea -> ea.getKey())
				.collect(Collectors.toList());
		return names.stream().flatMap(ea -> listVersions(ea).stream()).collect(Collectors.toList());		
	}
	
	public PositionableReadStream openPositionableReadStream(String key) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, null), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, String versionId, long position) {
		return new PositionableReadStream(this, key, getObjectMetadata(key, versionId), 0L, DEFAULT_BUFFER_SIZE);
	}

	public PositionableReadStream openPositionableReadStream(String key, ObjectMetadata metadata, long position) {
		return new PositionableReadStream(this, key, metadata, position, DEFAULT_BUFFER_SIZE);
	}

	public ObjectMetadata getObjectMetadata(String key, String versionId) {
		return s3Client.getObjectMetadata(new GetObjectMetadataRequest(bucketName, key, versionId));
	}
	
	public boolean getRange(String file, long position, int[] buffer) {
		GetObjectRequest request = new GetObjectRequest(bucketName, file).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(String file, String versionId, long position, int[] buffer) {		
		GetObjectRequest request = new GetObjectRequest(bucketName, file, versionId).withRange(position,
	    		position + buffer.length - 1);
		return getRange(request, buffer);
	}
	
	public boolean getRange(GetObjectRequest request, int[] buffer) {		
		try(S3Object s3Object = s3Client.getObject(request)) {
			int index = 0;
			S3ObjectInputStream stream = s3Object.getObjectContent();
			int next;
			while((next = stream.read()) != -1)
				buffer[index++] = next;
			return true;
		} catch (Exception e) {
			return false;
		}
	}

	public String getVersionId(String key) {
		// TODO Auto-generated method stub
		return null;
	}


}