package com.mission.base.s3;

import java.io.InputStream;
import java.io.StringWriter;
import java.util.Scanner;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.amazonaws.services.s3.model.ObjectMetadata;

public class PositionableReadStream extends InputStream {
	protected int maxBufferSize;
	protected final S3ClientConnector clientConnector;
	protected Scanner scanner; 
	protected int[] buffer; 
	protected long bufferStartPosition; 
	protected long bufferEndPosition; 
	protected final String key;
	protected final ObjectMetadata metadata;
	protected long position = 0;
	protected long objectLength;

    public PositionableReadStream(S3ClientConnector clientConnector, String key, ObjectMetadata metadata, long position, int maxBufferSize) {
        this.clientConnector = clientConnector;
        this.key = key;
        this.metadata = metadata;
        this.objectLength = getLength();
        this.position = position;
        this.maxBufferSize = maxBufferSize; 
        this.buffer = new int[maxBufferSize];
        getNextPage(position);
    }

    @Override
    public int read() {
        if ((position >= objectLength) 
        		|| ((position > bufferEndPosition) 
        		|| (position < bufferStartPosition)
        	&& !getNextPage(position)))
			return -1;
        int b = buffer[(int)(position - bufferStartPosition)];
        position++;
        return b;
    }

    public boolean getNextPage(long position) { 
    	bufferStartPosition = position;
    	bufferEndPosition = bufferStartPosition + buffer.length - 1;
    	return clientConnector.getRange(key, metadata.getVersionId(), position, buffer);
    }
    
    public String getContents() { 
    	long currentPosition = position;
    	position = 0;
    	StringWriter writer = new StringWriter();
    	int next;
    	while((next = read()) != -1)
    		writer.write(next);
    	position = currentPosition;
    	return writer.toString();
    }
    
    public PositionableReadStream clone() {
    	return clientConnector.openPositionableReadStream(key, metadata, position);
    }
    
    public long indexOf(String str) {
    	long currentPosition = position;
    	int[] chars = str.chars().toArray();
    	while(!nextMatches(chars)) 
    		if(read() == -1) {
    	    	position = currentPosition;
    			return -1;
    		}
    	long ans = position;
    	position = currentPosition;
    	return ans;
    }
    
    public boolean nextMatches(int[] chars) {
    	int charsLastIndex = chars.length - 1;
    	long endReadPosition = position + charsLastIndex;
    	if((endReadPosition > bufferEndPosition)
    			&& !getNextPage(position))
    		return false;   		
    	long currentPosition = position;
    	for(int i = 0; i <= charsLastIndex; i++)
    		if(chars[i] != read()) {
    	    	position = currentPosition;
    			return false;
    		}
    	position = currentPosition;
    	return true;
    }
    
    public String peekFor(long length) {
    	if((position + length > bufferEndPosition)
    			&& !getNextPage(position))
    		return null;   		
    	long currentPosition = position;
    	String ans = readTo(position + length);
    	position = currentPosition;
    	return ans;
    }
    
    public String read(int length) {
    	return readTo(position + length);
    }
    
    public String readTo(long endPosition) {
    	StringWriter writer = new StringWriter();
    	long end = Math.min(endPosition, objectLength - 1);
    	int length = (int)(end - position);
    	if(length < 0)
    		return null;
    	for(int i = 0; i < length; i++)
    		writer.write(read());
    	return writer.toString();
    }
    
    public IntStream intStream() { 
    	return IntStream.generate(() -> read()).limit(objectLength - position);      
    } 
    
    public Stream<Character> characterStream() {
    	return (Stream<Character>) IntStream.generate(() -> read())
    			.limit(objectLength - position)
    			.mapToObj(i -> (char)i);     
    }
    
    public Scanner scanner() {
    	if(scanner == null)
    		scanner = new Scanner(this); 
    	return scanner;
    }
    
    public Stream<String> stringStream() {
    	final Scanner scanner = scanner();
    	return Stream.generate(() -> scanner.next());     
    }

	public long getPosition() {
		return position;
	}

	public void setPosition(long position) {
		this.position = position;
	}

	public int occurrences(String string) {
		int count = 0;
		int[] chars = string.chars().toArray();
		if(nextMatches(chars))
			count++;
		while(read() != -1)
			if(nextMatches(chars))
				count++;
		return count;
	}

	public String getVersionId() {
		return metadata.getVersionId();
	}
	
	public long getLength() {
		return metadata.getContentLength();
	}
    
    
    
}