package pl.kro9bits.customizable_progress_bar;

import android.content.Context;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.text.NumberFormat;
import java.util.Locale;

/**
 * Created by mareksniezek on 03.11.2017.
 */

public class CustomizableProgressBar extends RelativeLayout{

    public enum RightTextMode{
        CUSTOM, PROGRESS_PERCENTAGE, PROGRESS_VALUE, PROGRESS_VALUE_CURRENCY
    }

    private RightTextMode rightTextMode = RightTextMode.CUSTOM;

    private CustomProgressBar progressBar;

    private TextView text_left;
    private TextView text_right;

    private ImageView icon_left;
    private ImageView icon_right;

    public CustomizableProgressBar(Context context) {
        super(context);
        init();
    }

    public CustomizableProgressBar(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init();

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.CustomizableProgressBar);
        parseAttributes(a);
    }

    public CustomizableProgressBar(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();

        TypedArray a = getContext().obtainStyledAttributes(attrs, R.styleable.CustomizableProgressBar, defStyleAttr, 0);
        parseAttributes(a);
    }

    /**
     * Ustawia zawartość i widoczność lewej etykiety
     * @param text Tekst do wyświetlenia. Jeśli null lub pusty etykieta zostaje ukryte.
     */
    public void setTextLeft(String text){
        if (text == null || (text != null && text.trim().length() == 0)){
            text_left.setVisibility(GONE);
            text_left.setText("");
        } else {
            text_left.setVisibility(VISIBLE);
            text_left.setText(text);
        }
    }

    /**
     * Ustawia zawartość i widoczność prawej etykiety
     * @param text Tekst do wyświetlenia. Jeśli null lub pusty etykieta zostaje ukryte.
     */
    public void setTextRight(String text){

        if (text == null || (text != null && text.trim().length() == 0)){
            text_right.setVisibility(GONE);
            text_right.setText("");
        } else {
            text_right.setVisibility(VISIBLE);
            text_right.setText(text);
        }
    }

    public void setIconLeftSrc(int resourceId){
        try {
            getResources().getResourceName(resourceId);
        } catch (Resources.NotFoundException e){
            return;
        }

        icon_left.setImageResource(resourceId);
        icon_left.setVisibility(VISIBLE);
    }

    public void setIconRightSrc(int resourceId){
        try {
            getResources().getResourceName(resourceId);
        } catch (Resources.NotFoundException e){
            return;
        }

        icon_right.setImageResource(resourceId);
        icon_right.setVisibility(VISIBLE);
    }

    public void setProgress(int progress){
        progressBar.setProgress(progress);
        refreshRightText();
    }

    public void setMax(int max){
        progressBar.setMax(max);
        refreshRightText();
    }

    public void setSecondaryProgress(int secondaryProgress){
        progressBar.setSecondaryProgress(secondaryProgress);
    }

    public void setBackgroundColor(int color){
        progressBar.setBackgroundColor(color);
    }

    public void setProgressColor(int color){
        progressBar.setProgressColor(color);
    }

    public void setSecondaryProgressColor(int color){
        progressBar.setSecondaryProgressColor(color);
    }

    public TextView getText_left(){
        return text_left;
    }

    public TextView getText_right(){
        return text_right;
    }

    public int getMax(){
        return progressBar.getMax();
    }

    public int getProgress(){
        return progressBar.getProgress();
    }

    public void setRightTextMode(RightTextMode rightTextMode) {
        this.rightTextMode = rightTextMode;
    }

    private void init(){
        View view = inflate(getContext(), R.layout.customizable_progress_bar, this);

        progressBar = view.findViewById(R.id.leo_progress_bar_progres_bar);

        text_left = view.findViewById(R.id.leo_progress_bar_text_left);
        text_right = view.findViewById(R.id.leo_progress_bar_text_right);

        icon_left = view.findViewById(R.id.leo_progress_bar_icon_left);
        icon_right = view.findViewById(R.id.leo_progress_bar_icon_right);
    }

    private void parseAttributes(TypedArray array){
        progressBar.setProgressColor(array.getColor(R.styleable.CustomizableProgressBar_progressColor, 0));
        progressBar.setBackgroundColor(array.getColor(R.styleable.CustomizableProgressBar_backgroundColor, 0));
        progressBar.setSecondaryProgressColor(array.getColor(R.styleable.CustomizableProgressBar_secondaryProgressColor, 0));

        progressBar.setProgress(array.getInteger(R.styleable.CustomizableProgressBar_progress, 0));
        progressBar.setSecondaryProgress(array.getInteger(R.styleable.CustomizableProgressBar_secondaryProgress, 0));
        progressBar.setMax(array.getInteger(R.styleable.CustomizableProgressBar_max, 0));

        switch (array.getInt(R.styleable.CustomizableProgressBar_rounding, 0)){
            case 0: progressBar.setProgressDrawable(getResources().getDrawable(R.drawable.custom_progress_bar_horizontal_0)); break;
            case 5: progressBar.setProgressDrawable(getResources().getDrawable(R.drawable.custom_progress_bar_horizontal_5dip)); break;
            case 10: progressBar.setProgressDrawable(getResources().getDrawable(R.drawable.custom_progress_bar_horizontal_10dip)); break;
            case 25: progressBar.setProgressDrawable(getResources().getDrawable(R.drawable.custom_progress_bar_horizontal_25dip)); break;
            case 50: progressBar.setProgressDrawable(getResources().getDrawable(R.drawable.custom_progress_bar_horizontal_50dip)); break;
        }

        setTextLeft(array.getString(R.styleable.CustomizableProgressBar_text_left));
        setTextRight(array.getString(R.styleable.CustomizableProgressBar_text_right));

        setIconLeftSrc(array.getResourceId(R.styleable.CustomizableProgressBar_icon_left, -1));
        setIconRightSrc(array.getResourceId(R.styleable.CustomizableProgressBar_icon_right, -1));

        switch (array.getInt(R.styleable.CustomizableProgressBar_right_text_mode, 0)){
            case 0: rightTextMode = RightTextMode.CUSTOM; break;
            case 1: rightTextMode = RightTextMode.PROGRESS_PERCENTAGE; break;
            case 2: rightTextMode = RightTextMode.PROGRESS_VALUE; break;
            case 3: rightTextMode = RightTextMode.PROGRESS_VALUE_CURRENCY; break;
        }

        refreshRightText();
        array.recycle();
    }

    private void refreshRightText(){
        switch (rightTextMode){
            case PROGRESS_PERCENTAGE: setTextRight(Integer.toString((int)(progressBar.getProgress() * 100 / progressBar.getMax())) + " %"); break;
            case PROGRESS_VALUE: setTextRight(Integer.toString(progressBar.getProgress())); break;
            case PROGRESS_VALUE_CURRENCY: setTextRight(NumberFormat.getCurrencyInstance(new Locale("PL", "pl")).format(progressBar.getProgress())); break;
        }
    }
}
