package com.kontakt.sdk.android.common.util;

import com.kontakt.sdk.android.common.interfaces.SDKFunction;
import com.kontakt.sdk.android.common.interfaces.SDKThrowableFunction;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

/**
 * JSON utility methods.
 */
public final class JSONUtils {

    private JSONUtils() {
    }

    private static final JSONArray EMPTY_JSON_ARRAY = new JSONArray();

    /**
     * Checks whether JSONObject contains specified key with non-null value.
     *
     * @param jsonObject the json object
     * @param key        the key
     * @return the boolean
     */
    public static boolean hasJSONKey(final JSONObject jsonObject, final String key) {
        return jsonObject.has(key) && !jsonObject.isNull(key);
    }

    /**
     * Gets string value bound with specified key. If the value is null, default value is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the string
     */
    public static String getString(final JSONObject jsonObject, final String key, final String defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getString(key) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Gets int value bound with specified key. If the value is null, default int value is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the int
     */
    public static int getInt(final JSONObject jsonObject, final String key, final int defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getInt(key) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Gets long value bound with specified key. If the value is null, default long value is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the long
     */
    public static long getLong(final JSONObject jsonObject, final String key, final long defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getLong(key) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Gets double value bound with specified key. If the value is null, default double value is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the double
     */
    public static double getDouble(final JSONObject jsonObject, final String key, final double defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getInt(key) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Gets UUID value bound with specified key. If the value is null, default UUID is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the uUID
     */
    public static UUID getUUID(final JSONObject jsonObject, final String key, final UUID defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? UUID.fromString(jsonObject.getString(key)) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Gets UUID value bound with specified key or null if the key does not exist.
     *
     * @param jsonObject the json object
     * @param key        the key
     * @return the uUID or null
     */
    public static UUID getUUIDOrNull(final JSONObject jsonObject, final String key) {
        return getUUID(jsonObject, key, null);
    }

    /**
     * Gets string or null.
     *
     * @param jsonObject the json object
     * @param key        the key
     * @return the string or null
     */
    public static String getStringOrNull(final JSONObject jsonObject, final String key) {
        return getString(jsonObject, key, null);
    }

    /**
     * Gets boolean value bound with specified key. If the value does not exist or is null, default value is returned.
     *
     * @param jsonObject   the json object
     * @param key          the key
     * @param defaultValue the default value
     * @return the boolean
     */
    public static boolean getBoolean(final JSONObject jsonObject, final String key, final boolean defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getBoolean(key) : defaultValue;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    public static Boolean getBooleanBoxed(final JSONObject jsonObject, final String key) {
        try {
            Object value = jsonObject.get(key);
            if (value == null) {
                return null;
            }
            if (value instanceof Boolean) {
                return (Boolean) value;
            }
            return null;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    public static JSONArray getJSONArray(final JSONObject jsonObject, final String key, final JSONArray defaultValue) {
        try {
            return hasJSONKey(jsonObject, key) ? jsonObject.getJSONArray(key) : defaultValue;
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    public static JSONObject getJSONObject(final JSONArray sourceObject, final int index) {
        try {
            return sourceObject.getJSONObject(index);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    public static String getJSONArrayElement(final JSONArray sourceArray, final int index) {
        try {
            return sourceArray.getString(index);
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Transform JSONObject to desired type.
     *
     * @param <T>               the type parameter
     * @param jsonObject        the json object
     * @param transformFunction the throwable function
     * @return the instance of desired type
     * @throws IllegalStateException in case of failure
     */
    public static <T> T transformOrThrow(final JSONObject jsonObject, final SDKThrowableFunction<JSONObject, T, Exception> transformFunction) {
        try {
            return transformFunction.apply(jsonObject);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Transform JSONArray to desired type.
     *
     * @param <T>               the type parameter
     * @param jsonArray         the json array
     * @param transformFunction the throwable function
     * @return the t
     * @throws IllegalStateException in case of failure
     */
    public static <T> T transformOrThrow(final JSONArray jsonArray, final SDKThrowableFunction<JSONArray, T, Exception> transformFunction) {
        try {
            return transformFunction.apply(jsonArray);
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Transform to list or return empty.
     *
     * @param <T>        the type parameter
     * @param jsonObject the json object
     * @param tag        the tag
     * @param function   the function
     * @return the list
     */
    public static <T> List<T> transformToListOrReturnEmpty(final JSONObject jsonObject, final String tag, final SDKFunction<JSONObject, T> function) {
        try {
            final JSONArray jsonArray = hasJSONKey(jsonObject, tag) ? jsonObject.getJSONArray(tag) : EMPTY_JSON_ARRAY;

            if (jsonArray.length() == 0) {
                return Collections.emptyList();
            }

            List<T> list = new ArrayList<T>();
            for (int i = 0, size = jsonArray.length(); i < size; i++) {
                list.add(function.apply(jsonArray.getJSONObject(i)));
            }

            return list;
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Transform JSONObject's property accessible under specified entry to desired instance or return null.
     *
     * @param <T>             the type parameter
     * @param jsonObject      the json object
     * @param jsonEntry       the json entry
     * @param convertFunction the convert function
     * @return the t
     */
    public static <T> T transformOrReturnNull(JSONObject jsonObject, String jsonEntry, final SDKFunction<JSONObject, T> convertFunction) {
        try {
            return hasJSONKey(jsonObject, jsonEntry) ? convertFunction.apply(jsonObject.getJSONObject(jsonEntry)) : null;
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }
}
