package com.kontakt.sdk.android.common.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Utility methods for Http interaction.
 */
public final class HttpUtils {
    private HttpUtils() {
    }

    /**
     * Http Status SC_BAD_REQUEST.
     */
    public static final int SC_BAD_REQUEST = 400;
    /**
     * Http Status SC_CONFLICT.
     */
    public static final int SC_CONFLICT = 409;
    /**
     * Http Status SC_CREATED.
     */
    public static final int SC_CREATED = 201;
    /**
     * Http Status SC_FORBIDDEN.
     */
    public static final int SC_FORBIDDEN = 403;
    /**
     * Http Status SC_INTERNAL_SERVER_ERROR.
     */
    public static final int SC_INTERNAL_SERVER_ERROR = 500;
    /**
     * Http Status SC_NOT_FOUND.
     */
    public static final int SC_NOT_FOUND = 404;
    /**
     * Http Status SC_NOT_MODIFIED.
     */
    public static final int SC_NOT_MODIFIED = 304;
    /**
     * Http Status SC_OK.
     */
    public static final int SC_OK = 200;

    public static final int SC_NO_CONTENT = 204;
    /**
     * Http Status SC_SERVICE_UNAVAILABLE.
     */
    public static final int SC_SERVICE_UNAVAILABLE = 503;
    /**
     * Http Status SC_UNAUTHORIZED.
     */
    public static final int SC_UNAUTHORIZED = 401;
    /**
     * Http Status SC_UNSUPPORTED_MEDIA_TYPE.
     */
    public static final int SC_UNSUPPORTED_MEDIA_TYPE = 415;
    /**
     * Http Status SC_SEE_OTHER.
     */
    public static final int SC_SEE_OTHER = 303;
    /**
     * Http Status SC_METHOD_NOT_ALLOWED.
     */
    public static final int SC_METHOD_NOT_ALLOWED = 405;
    /**
     * Http Status SC_UNPROCESSABLE_ENTITY.
     */
    public static final int SC_UNPROCESSABLE_ENTITY = 422;
    /**
     * Http Status SC_BAD_GATEWAY.
     */
    public static final int SC_BAD_GATEWAY = 502;
    /**
     * Http Status SC_GATEWAY_TIMEOUT.
     */
    public static final int SC_GATEWAY_TIMEOUT = 504;

    /**
     * Creates NameValuePair where the name is a first String parameter and the specified collection provides
     * values converted to String representation.
     *
     * @param <T>           the type parameter
     * @param argumentName  the argument name
     * @param argCollection the arg collection
     * @return the name value pair [ ]
     */
    public static <T> List<Map.Entry<String, String>> toUrlParameterList(final String argumentName, final Collection<T> argCollection) {
        SDKPreconditions.checkNotNullOrEmpty(argCollection, "Arguments array is empty or null.");
        SDKPreconditions.checkNotNullOrEmpty(argumentName, "Argument name is null or empty");

        List<Map.Entry<String, String>> entries = new ArrayList<Map.Entry<String, String>>(argCollection.size());

        for (T element : argCollection) {
            entries.add(new UrlParameter(argumentName, (element instanceof String) ? (String) element : element.toString()));
        }

        return entries;
    }

    public static <T> Map.Entry<String, String> toUrlParameter(final String argumentName, final String argumentValue) {
        SDKPreconditions.checkNotNull(argumentName);
        SDKPreconditions.checkNotNull(argumentValue);

        return new UrlParameter(argumentName, argumentValue);
    }

    private static class UrlParameter implements Map.Entry<String, String> {

        private final String key;

        private final String value;

        private final int hashCode;

        private UrlParameter(String key, String value) {
            this.key = key;
            this.value = value;
            this.hashCode = HashCodeBuilder.init()
                    .append(key)
                    .append(value)
                    .build();
        }

        @Override
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }

            if (o == null || !(o instanceof UrlParameter)) {
                return false;
            }

            UrlParameter urlParameter = (UrlParameter) o;

            return this.key.equals(urlParameter.key) && this.value.equals(urlParameter.value);
        }

        @Override
        public int hashCode() {
            return hashCode;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getValue() {
            return value;
        }

        @Override
        public String setValue(String object) {
            throw new UnsupportedOperationException();
        }
    }
}
