package com.kontakt.sdk.android.common.util;

import android.util.SparseArray;

import java.nio.ByteBuffer;
import java.util.regex.Pattern;

/**
 * Utility methods for Eddystone devices.
 */
public final class EddystoneUtils {
    private static final SparseArray<String> URL_SCHEME_PREFIXES;

    private static final SparseArray<String> URL_SCHEME_SUFFIXES;

    private static final int MASK = 0x00FF;
    private static final int SUFFIX_MASK = 0x000F;

    private static final String NULL_CHAR = "\0";
    private static final String HEX_STRING_FORMAT = "%02x";

    private static final int LOWEST_ASCII_PRINTABLE_VALUE = 32;
    private static final int HIGHEST_ASCII_PRINTABLE_VALUE = 127;

    private static final char[] HEX = "0123456789ABCDEF".toCharArray();

    private static final Pattern HEX_PATTERN = Pattern.compile("[0-9a-fA-F]+");

    static {
        URL_SCHEME_PREFIXES = new SparseArray<String>(4);

        URL_SCHEME_PREFIXES.put(0x00, "http://www.");
        URL_SCHEME_PREFIXES.put(0x01, "https://www.");
        URL_SCHEME_PREFIXES.put(0x02, "http://");
        URL_SCHEME_PREFIXES.put(0x03, "https://");

        URL_SCHEME_SUFFIXES = new SparseArray<String>(14);

        URL_SCHEME_SUFFIXES.put(0x00, ".com/");
        URL_SCHEME_SUFFIXES.put(0x01, ".org/");
        URL_SCHEME_SUFFIXES.put(0x02, ".edu/");
        URL_SCHEME_SUFFIXES.put(0x03, ".net/");
        URL_SCHEME_SUFFIXES.put(0x04, ".info/");
        URL_SCHEME_SUFFIXES.put(0x05, ".biz/");
        URL_SCHEME_SUFFIXES.put(0x06, ".gov/");
        URL_SCHEME_SUFFIXES.put(0x07, ".com");
        URL_SCHEME_SUFFIXES.put(0x08, ".org");
        URL_SCHEME_SUFFIXES.put(0x09, ".edu");
        URL_SCHEME_SUFFIXES.put(0x0a, ".net");
        URL_SCHEME_SUFFIXES.put(0x0b, ".info");
        URL_SCHEME_SUFFIXES.put(0x0c, ".biz");
        URL_SCHEME_SUFFIXES.put(0x0d, ".gov");
    }

    private EddystoneUtils() {

    }

    /**
     * Transforms bytes array into String representation.
     *
     * @param value the value
     * @return the string
     */
    public static String toStringFromHex(byte[] value) {
        StringBuilder builder = new StringBuilder();
        for (byte aValue : value) {
            int b = (aValue & MASK);
            builder.append(String.format(HEX_STRING_FORMAT, b));
        }
        return builder.toString();
    }

    /**
     * Converts array of bytes to hexed string.
     *
     * @param bytes the bytes
     * @return the string
     */
    public static String toHexString(byte[] bytes) {
        char[] chars = new char[bytes.length * 2];
        for (int i = 0; i < bytes.length; i++) {
            int c = bytes[i] & 0xFF;
            chars[i * 2] = HEX[c >>> 4];
            chars[i * 2 + 1] = HEX[c & 0x0F];
        }
        return new String(chars).toLowerCase();
    }

    /**
     * Creates String with with specific length aligned to the end with
     * filled with "0" from the beginning.
     *
     * @param arg          the arg
     * @param targetLength the target length
     * @return the string
     */
    public static String toAlignedString(String arg, int targetLength) {
        int length = arg.getBytes().length;
        if (length < targetLength) {
            StringBuilder builder = new StringBuilder();
            for (int i = length; i < targetLength; i++) {
                builder.append(NULL_CHAR);
            }
            builder.append(arg);
            return builder.toString();
        }

        return arg;
    }


    /**
     * Provides url from byte array.
     *
     * @param values the values
     * @return the url
     */
    public static String deserializeUrl(byte[] values) {
        if (values == null || values.length == 0) {
            return null;
        }
        String prefix = extractPrefix(values[0]);
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(prefix);

        for (int i = 1; i < values.length; i++) {
            char ch = (char) values[i];
            if (isAsciiPrintable(ch)) {
                urlBuilder.append(ch);
            } else {
                String suffixFromByte = getExtractSuffixFromByte(values[i]);
                if (suffixFromByte != null) {
                    urlBuilder.append(suffixFromByte);
                }
            }
        }
        return urlBuilder.toString();
    }

    /**
     * Serializes url into Eddystone-based byte array format.
     *
     * @param url the url
     * @return the byte array
     */
    public static byte[] serializeUrl(String url) {
        isAllASCII(url);
        //prefix
        for (int i = 0; i < URL_SCHEME_PREFIXES.size(); i++) {
            int key = URL_SCHEME_PREFIXES.keyAt(i);
            String prefix = URL_SCHEME_PREFIXES.get(key);
            int prefixSectionIndexOf = url.indexOf(prefix);
            if (prefixSectionIndexOf != -1) {
                byte[] prefixInBytes = ByteBuffer.allocate(4)
                        .putInt(key)
                        .array();

                url = url.replace(prefix, new String(prefixInBytes, 3, 1));
            }
        }

        //suffix
        for (int i = 0; i < URL_SCHEME_SUFFIXES.size(); i++) {
            int key = URL_SCHEME_SUFFIXES.keyAt(i);
            String suffix = URL_SCHEME_SUFFIXES.get(key);
            int suffixSectionIndexOf = url.indexOf(suffix);
            if (suffixSectionIndexOf != -1) {
                byte[] suffixInBytes = ByteBuffer.allocate(4)
                        .putInt(key)
                        .array();

                url = url.replace(suffix, new String(suffixInBytes, 3, 1));
            }
        }
        return url.getBytes();
    }

    /**
     * Returns human readable representation from hexed String.
     * Input String must be serialized in Eddystone format
     * From hexed url to url.
     *
     * @param hex the hex
     * @return the string
     */
    public static String fromHexedUrlToUrl(String hex) {
        byte[] decode = decodeHexedUrl(hex.trim());
        return new String(decode).trim();
    }

    /**
     * Converts hexed string to byte array
     *
     * @param s the string
     * @return byte array
     */
    public static byte[] hexStringToByteArray(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len - 1; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
                    + Character.digit(s.charAt(i + 1), 16));
        }
        return data;
    }

    public static boolean isStringOnlyHex(String input) {
        return HEX_PATTERN.matcher(input).matches();
    }

    private static byte[] decodeHexedUrl(String hex) {

        String[] list = splitStringEvery(hex, 2);

        ByteBuffer buffer = ByteBuffer.allocate(Constants.Eddystone.URL_MAX_BYTES_LENGTH * 2);
        String firstByte = list[0];
        String prefix = extractPrefix(Byte.parseByte(firstByte, 16));

        if (prefix != null) {
            int prefixLength = prefix.length();
            for (int i = 0; i < prefixLength; i++) {
                buffer.put((byte) prefix.charAt(i));
            }
        }

        int length = list.length;

        for (int i = 1; i < length; i++) {
            byte parseByte = Byte.parseByte(list[i], 16);
            boolean asciiPrintable = isAsciiPrintable((char) parseByte);
            if (asciiPrintable) {
                buffer.put(parseByte);
            } else {
                String suffixFromByte = getExtractSuffixFromByte(parseByte);
                if (suffixFromByte != null) {
                    int suffixLength = suffixFromByte.length();
                    for (int j = 0; j < suffixLength; j++) {
                        buffer.put((byte) suffixFromByte.charAt(j));
                    }
                }
            }
        }

        return buffer.array();
    }

    private static String[] splitStringEvery(String s, int interval) {
        int arrayLength = (int) Math.ceil(((s.length() / (double) interval)));
        String[] result = new String[arrayLength];

        int j = 0;
        int lastIndex = result.length - 1;
        for (int i = 0; i < lastIndex; i++) {
            result[i] = s.substring(j, j + interval);
            j += interval;
        }
        result[lastIndex] = s.substring(j);

        return result;
    }

    private static String extractPrefix(byte prefixByte) {
        int prefixArrayValue = (prefixByte & MASK);
        return URL_SCHEME_PREFIXES.get(prefixArrayValue);
    }

    private static String getExtractSuffixFromByte(byte b) {
        return URL_SCHEME_SUFFIXES.get(b & SUFFIX_MASK);
    }

    private static boolean isAsciiPrintable(char ch) {
        return ch >= LOWEST_ASCII_PRINTABLE_VALUE &&
                ch < HIGHEST_ASCII_PRINTABLE_VALUE;
    }

    public static void isAllASCIIPrintable(String input) {
        boolean isASCII = true;
        for(int i = 0; i < input.length(); i++){
            char c = input.charAt(i);
            if(!isAsciiPrintable(c)){
                isASCII = false;
                break;
            }
        }
        if(!isASCII) {
            throw new IllegalArgumentException("Provided input must be all ASCII printable");
        }
    }

    private static void isAllASCII(String input) {
        boolean isASCII = true;
        for (int i = 0; i < input.length(); i++) {
            int c = input.charAt(i);
            if (c > 0x7F) {
                isASCII = false;
                break;
            }
        }
        if (!isASCII) {
            throw new IllegalArgumentException("Provided url must be ASCII");
        }
    }

}
