package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

/**
 * Venue acts as Region.
 * Additionally it is a describable container for Beacons which can be signified
 * with image cover.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Venue.Builder}.
 */
public class Venue extends AbstractModel implements IVenue {

    private final UUID id;

    private final String name;

    private final String description;

    private final int beaconsCount;

    private final String coverType;

    private final List<IDevice> devices;

    private final String imageUrl;

    private final UUID managerId;

    private final int hashCode;

    private final String longitude;

    private final String latitude;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Parcelable.Creator<Venue> CREATOR = new Parcelable.Creator<Venue>() {
        public Venue createFromParcel(Parcel input) {
            final Builder builder = new Builder();
            final Bundle bundle = input.readBundle(getClass().getClassLoader());

            final int beaconsListSize = input.readInt();

            for (int index = 0; index < beaconsListSize; index++) {
                Beacon beacon = Beacon.CREATOR.createFromParcel(input);
                builder.addDevice(beacon);
            }

            return builder
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setName(bundle.getString(Constants.Venue.NAME))
                    .setDescription(bundle.getString(Constants.Venue.DESCRIPTION))
                    .setCoverType(bundle.getString(Constants.Venue.COVER_TYPE))
                    .setDevicesCount(bundle.getInt(Constants.Venue.DEVICES_COUNT))
                    .setImageUrl(bundle.getString(Constants.Venue.IMAGE))
                    .setManagerId((UUID) bundle.getSerializable(Constants.Venue.MANAGER_ID))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setLatitude(bundle.getString(Constants.Venue.LATITUDE))
                    .setLongitude(bundle.getString(Constants.Venue.LONGITUDE))
                    .build();
        }

        public Venue[] newArray(int size) {
            return new Venue[size];
        }
    };

    /**
     * Instantiates a new Venue model.
     *
     * @param builder the Venue Builder
     */
    private Venue(Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.coverType = builder.coverType;
        this.beaconsCount = builder.beaconsCount;
        this.imageUrl = builder.imageUrl;
        this.devices = Collections.unmodifiableList(builder.beacons);
        this.managerId = builder.managerId;
        this.latitude = builder.lat;
        this.longitude = builder.lng;

        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .append(name)
                .append(description)
                .append(coverType)
                .append(beaconsCount)
                .append(imageUrl)
                .append(devices)
                .append(managerId)
                .append(latitude)
                .append(longitude)
                .build();
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof Venue)) {
            return false;
        }

        final Venue venue = (Venue) o;

        return SDKEqualsBuilder.start()
                .equals(id, venue.id)
                .equals(description, venue.description)
                .equals(coverType, venue.coverType)
                .equals(beaconsCount, venue.beaconsCount)
                .equals(imageUrl, venue.imageUrl)
                .equals(devices, venue.devices)
                .equals(managerId, venue.managerId)
                .equals(latitude, venue.latitude)
                .equals(longitude, venue.longitude)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getCoverType() {
        return coverType;
    }

    @Override
    public List<IDevice> getDevices() {
        return devices;
    }

    @Override
    public UUID getManagerId() {
        return managerId;
    }

    @Override
    public int getDevicesCount() {
        return beaconsCount;
    }

    @Override
    public String getImageUrl() {
        return imageUrl;
    }

    @Override
    public String getLatitude() {
        return latitude;
    }

    @Override
    public String getLongitude() {
        return longitude;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putSerializable(Constants.Venue.MANAGER_ID, managerId);
        bundle.putString(Constants.Venue.NAME, name);
        bundle.putString(Constants.Venue.DESCRIPTION, description);
        bundle.putString(Constants.Venue.COVER_TYPE, coverType);
        bundle.putInt(Constants.Venue.DEVICES_COUNT, beaconsCount);
        bundle.putString(Constants.Venue.IMAGE, imageUrl);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putString(Constants.Venue.LONGITUDE, longitude);
        bundle.putString(Constants.Venue.LATITUDE, latitude);
        dest.writeBundle(bundle);

        dest.writeInt(devices.size());

        for (IDevice beacon : devices) {
            beacon.writeToParcel(dest, PARCELABLE_FLAG_EXCLUDE_VENUE);
        }
    }

    /**
     * Creates new Venue instance from JSON content and File data.
     * The raw JSON content can be obtained via REST Client.
     *
     * @param jsonObject the json object
     * @return the venue
     */
    public static Venue from(final JSONObject jsonObject) {
        try {
            final List<IDevice> beaconsList = new ArrayList<IDevice>();

            if (jsonObject.has(Constants.Venue.DEVICES)) {
                JSONArray beaconsArray = jsonObject.getJSONArray(Constants.Venue.DEVICES);

                for (int index = 0, limit = beaconsArray.length(); index < limit; index++) {
                    beaconsList.add(Beacon.from(beaconsArray.getJSONObject(index)));
                }
            }

            return new Builder()
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .setManagerId(JSONUtils.getUUID(jsonObject, Constants.Venue.MANAGER_ID, null))
                    .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.NAME))
                    .setDescription(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.DESCRIPTION))
                    .setCoverType(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.COVER_TYPE))
                    .setDevicesCount(JSONUtils.getInt(jsonObject, Constants.Venue.DEVICES_COUNT, 0))
                    .setImageUrl(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.IMAGE))
                    .addDevices(beaconsList)
                    .setLatitude(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.LATITUDE))
                    .setLongitude(JSONUtils.getStringOrNull(jsonObject, Constants.Venue.LONGITUDE))
                    .build();
        } catch (JSONException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Venue Builder.
     */
    public static final class Builder {
        private int databaseId;
        private UUID id;
        private String name;
        private String description;
        private String coverType;
        private int beaconsCount;
        private String imageUrl;
        private List<IDevice> beacons = new ArrayList<IDevice>();
        private UUID managerId;
        private String lat;
        private String lng;

        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the builder
         */
        public Builder setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the builder
         */
        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets description.
         *
         * @param description the description
         * @return the builder
         */
        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        /**
         * Sets cover type.
         *
         * @param coverType the cover type
         * @return the builder
         */
        public Builder setCoverType(String coverType) {
            this.coverType = coverType;
            return this;
        }

        /**
         * Sets devices count.
         *
         * @param beaconsCount the devices count
         * @return the builder
         */
        public Builder setDevicesCount(int beaconsCount) {
            this.beaconsCount = beaconsCount;
            return this;
        }

        /**
         * Sets image url.
         *
         * @param imageUrl the image url
         * @return the builder
         */
        public Builder setImageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
            return this;
        }

        /**
         * Add devices.
         *
         * @param beacons the devices
         * @return the builder
         */
        public Builder addDevices(final Collection<IDevice> beacons) {
            this.beacons.addAll(beacons);
            return this;
        }

        /**
         * Add beacon.
         *
         * @param beacon the beacon
         * @return the builder
         */
        public Builder addDevice(final IDevice beacon) {
            this.beacons.add(beacon);
            return this;
        }

        /**
         * Sets manager Id.
         *
         * @param managerId the manager Id
         * @return the builder
         */
        public Builder setManagerId(final UUID managerId) {
            this.managerId = managerId;
            return this;
        }

        /**
         * Sets longitude.
         *
         * @param lng the lng
         * @return the longitude
         */
        public Builder setLongitude(final String lng) {
            this.lng = lng;
            return this;
        }

        /**
         * Sets latitude.
         *
         * @param lat the lat
         * @return the latitude
         */
        public Builder setLatitude(final String lat) {
            this.lat = lat;
            return this;
        }

        /**
         * Builds Venue.
         *
         * @return the venue
         */
        public Venue build() {
            return new Venue(this);
        }
    }
}