package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.TimestampUtil;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.JSONUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Represents Api Object for secure configs
 * <p>
 * Can be retrieved from API as a list of pending configs for device and as well as a response from beacon to API
 * </p>
 */
public class SecureSingleConfig implements Parcelable {

    private static final String UNIQUE_ID = "uniqueId";
    private static final String CONFIG = "config";
    private static final String CONFIGS = "configs";

    private final String uniqueId;
    private final String secureMessage;
    private final long unixTimestamp;

    public static final Creator<SecureSingleConfig> CREATOR = new Creator<SecureSingleConfig>() {
        @Override
        public SecureSingleConfig createFromParcel(Parcel source) {
            return new SecureSingleConfig(source);
        }

        @Override
        public SecureSingleConfig[] newArray(int size) {
            return new SecureSingleConfig[size];
        }
    };

    /**
     * Constructor for SecureSingleConfig
     *
     * @param uniqueId      device unique id
     * @param secureMessage secure message from device or for device
     * @param unixTimestamp timestamp when applied to device
     */
    public SecureSingleConfig(String uniqueId, String secureMessage, long unixTimestamp) {
        this.uniqueId = uniqueId;
        this.secureMessage = secureMessage;
        this.unixTimestamp = unixTimestamp;
    }

    private SecureSingleConfig(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        this.uniqueId = bundle.getString(Constants.UNIQUE_ID);
        this.secureMessage = bundle.getString(Constants.SECURE_CONFIG);
        this.unixTimestamp = bundle.getLong(Constants.TIMESTAMP);
    }

    public String getUniqueId() {
        return uniqueId;
    }

    public String getSecureMessage() {
        return secureMessage;
    }

    public long getUnixTimestamp() {
        return unixTimestamp;
    }

    @Override
    public String toString() {
        return "SecureSingleConfig{" +
                "uniqueId='" + uniqueId + '\'' +
                ", secureMessage='" + secureMessage + '\'' +
                ", unixTimestamp=" + unixTimestamp +
                '}';
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(Constants.UNIQUE_ID, uniqueId);
        bundle.putString(Constants.SECURE_CONFIG, secureMessage);
        bundle.putLong(Constants.TIMESTAMP, unixTimestamp);
        dest.writeBundle(bundle);
    }

    /**
     * Creates SecureSingleConfig from JSONObject
     *
     * @param object the json object
     * @return SecureSingleConfig
     */
    public static final SecureSingleConfig fromJson(JSONObject object) {
        String uniqueId = JSONUtils.getStringOrNull(object, UNIQUE_ID);
        String config = JSONUtils.getStringOrNull(object, CONFIG);
        return new SecureSingleConfig(uniqueId, config, TimestampUtil.getTimestamp());
    }

    /**
     * Creates List of SecureSingleConfig from JSONObject
     *
     * @param object the json object
     * @return SecureSingleConfig
     */
    public static final List<SecureSingleConfig> fromListJson(JSONObject object) {
        JSONArray configs = JSONUtils.getJSONArray(object, CONFIGS, new JSONArray());
        List<SecureSingleConfig> secureConfigs = new ArrayList<SecureSingleConfig>();
        for (int i = 0; i < configs.length(); i++) {
            secureConfigs.add(fromJson(JSONUtils.getJSONObject(configs, i)));
        }
        return secureConfigs;
    }
}
