package com.kontakt.sdk.android.common.model;

import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import java.io.Serializable;
import java.util.UUID;

public class ResolvedId implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final String ID_SEPARATOR = ":";
    private final String deviceId;
    private final String uniqueId;
    private Integer hashCode;

    public ResolvedId(String deviceId, String uniqueId) {
        this.deviceId = deviceId;
        this.uniqueId = uniqueId;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof ResolvedId)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        ResolvedId resolvedId = (ResolvedId) o;
        return SDKEqualsBuilder.start()
                .equals(deviceId, resolvedId.deviceId)
                .equals(uniqueId, resolvedId.uniqueId)
                .result();
    }

    @Override
    public int hashCode() {
        if (hashCode == null) {
            hashCode = HashCodeBuilder.init()
                    .append(deviceId)
                    .append(uniqueId)
                    .build();
        }
        return hashCode;
    }

    public BeaconId getIBeaconId() {
        String[] splittedBeaconId = deviceId.split(ID_SEPARATOR);
        SDKPreconditions.checkArgument(splittedBeaconId.length == 3, "Incorrect iBeacon id format!");

        UUID proximity = UUID.fromString(splittedBeaconId[0]);
        int major = Integer.parseInt(splittedBeaconId[1]);
        int minor = Integer.parseInt(splittedBeaconId[2]);

        return BeaconId.of(proximity, major, minor);
    }

    public EddystoneUID getEddystoneUID() {
        String[] splittedEddystoneUID = deviceId.split(ID_SEPARATOR);
        SDKPreconditions.checkArgument(splittedEddystoneUID.length == 2, "Incorrect eddstone uid format!");

        String namespace = splittedEddystoneUID[0];
        String instanceId = splittedEddystoneUID[1];

        return EddystoneUID.of(namespace, instanceId);
    }

    public String getUniqueId() {
        return uniqueId;
    }

}
