package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Represents Proximity UUID. Consists of {@link UUID} as proximity and flag {@link ProximityUUID#shuffled} indicating is particular proximity shuffled or not
 */
public class ProximityUUID extends AbstractModel implements IProximityUUID, Parcelable {

    public static final Creator<ProximityUUID> CREATOR = new Creator<ProximityUUID>() {
        @Override
        public ProximityUUID createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());
            return new Builder()
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setProximityUUID(((UUID) bundle.getSerializable(Constants.Proximity.PROXIMITY_UUID)))
                    .setSecureProximityUUID((UUID) bundle.getSerializable(Constants.Proximity.SECURE_PROXIMITY_UUID))
                    .setShuffled(bundle.getBoolean(Constants.Proximity.SHUFFLED))
                    .setShared(bundle.getBoolean(Constants.Proximity.SHARED))
                    .build();
        }

        @Override
        public ProximityUUID[] newArray(int size) {
            return new ProximityUUID[size];
        }
    };

    private static final String PROXIMITY = "proximity";
    private static final String SHUFFLED = "shuffled";
    private static final String PROXIMITIES = "proximities";

    private final UUID proximityUUID;
    private final UUID secureProximityUUID;
    private final boolean shuffled;
    private final boolean shared;


    private ProximityUUID(Builder builder) {
        super(builder.databaseId);
        this.proximityUUID = builder.proximityUUID;
        this.secureProximityUUID = builder.secureProximityUUID;
        this.shuffled = builder.shuffled;
        this.shared = builder.shared;
    }


    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.Proximity.PROXIMITY_UUID, proximityUUID);
        bundle.putSerializable(Constants.Proximity.SECURE_PROXIMITY_UUID, secureProximityUUID);
        bundle.putBoolean(Constants.Proximity.SHUFFLED, shuffled);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putBoolean(Constants.Proximity.SHARED, shared);
        dest.writeBundle(bundle);
    }

    @Override
    public UUID getProximity() {
        return proximityUUID;
    }

    @Override
    public UUID getSecureProximity() {
        return secureProximityUUID;
    }

    @Override
    public boolean isShuffled() {
        return shuffled;
    }

    @Override
    public boolean isShared() {
        return shared;
    }

    public static IProximityUUID from(final JSONObject jsonObject) {

        UUID proximity = JSONUtils.getUUIDOrNull(jsonObject, "proximity");
        UUID secureProximity = JSONUtils.getUUIDOrNull(jsonObject, "secureProximity");
        boolean shuffled = JSONUtils.getBoolean(jsonObject, "shuffled", false);
        boolean shared = JSONUtils.getBoolean(jsonObject, "shared", false);

        return new ProximityUUID.Builder()
                .setShuffled(shuffled)
                .setProximityUUID(proximity)
                .setSecureProximityUUID(secureProximity)
                .setShared(shared)
                .build();
    }

    public static List<IProximityUUID> fromList(final JSONObject jsonObject) {
        try {
            List<IProximityUUID> proximities = new ArrayList<IProximityUUID>();

            JSONArray jsonArray = jsonObject.getJSONArray("proximities");

            for (int i = 0; i < jsonArray.length(); i++) {
                proximities.add(from(jsonArray.getJSONObject(i)));
            }
            return proximities;
        } catch (JSONException e) {
            throw new IllegalStateException("No proximities found");
        }
    }


    public static class Builder {
        private int databaseId;
        private UUID proximityUUID;
        private UUID secureProximityUUID;
        private boolean shuffled;
        private boolean shared;

        public Builder setDatabaseId(int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        public Builder setProximityUUID(UUID proximityUUID) {
            SDKPreconditions.checkNotNull(proximityUUID, "Proximity UUID cannot be null");
            this.proximityUUID = proximityUUID;
            return this;
        }

        public Builder setSecureProximityUUID(UUID secureProximityUUID) {
            SDKPreconditions.checkNotNull(secureProximityUUID, "Secure proximity UUID cannot be null");
            this.secureProximityUUID = secureProximityUUID;
            return this;
        }

        public Builder setShuffled(boolean shuffled) {
            this.shuffled = shuffled;
            return this;
        }

        public Builder setShared(boolean shared) {
            this.shared = shared;
            return this;
        }

        public ProximityUUID build() {
            return new ProximityUUID(this);
        }
    }
}
