package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONObject;

import java.util.UUID;

/**
 * Preset describes the predefined set of parameters that Beacon can be
 * configured with. Contrary to Configs, Profiles are not removed once the Beacon
 * device will be updated with parameters applied from them. They should be
 * interpreted as states in which Beacon devices are working. For example, you
 * could define "Long interval profile" where device interval period accepts the
 * the longest interval.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Preset.Builder}.
 */
public class Preset extends AbstractModel implements IPreset {

    private final UUID proximityUUID;
    private final int interval;
    private final int txPower;
    private final String name;
    private final String description;
    private final int hashCode;

    /**
     * Instantiates a new Preset.
     *
     * @param builder the builder
     */
    private Preset(Builder builder) {
        super(builder.databaseId);
        this.proximityUUID = builder.proximityUUID;
        this.interval = builder.interval;
        this.txPower = builder.txPower;
        this.name = builder.name;
        this.description = builder.description;

        this.hashCode = HashCodeBuilder.init()
                .append(proximityUUID)
                .append(interval)
                .append(txPower)
                .append(name)
                .append(description)
                .build();
    }


    @Override
    public UUID getProximityUUID() {
        return proximityUUID;
    }

    @Override
    public int getInterval() {
        return interval;
    }

    @Override
    public int getTxPower() {
        return txPower;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }

        if (obj == null || !(obj instanceof Preset)) {
            return false;
        }

        final Preset profile = (Preset) obj;

        return SDKEqualsBuilder.start()
                .equals(proximityUUID, profile.proximityUUID)
                .equals(interval, profile.interval)
                .equals(txPower, profile.txPower)
                .equals(name, profile.name)
                .equals(description, profile.description)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Parcelable.Creator<Preset> CREATOR = new Parcelable.Creator<Preset>() {
        @Override
        public Preset createFromParcel(Parcel source) {
            final Bundle bundle = source.readBundle(getClass().getClassLoader());

            return new Builder()
                    .setProximityUUID((UUID) bundle.getSerializable(Constants.Profile.PROXIMITY))
                    .setInterval(bundle.getInt(Constants.Profile.INTERVAL))
                    .setTxPower(bundle.getInt(Constants.Profile.TX_POWER))
                    .setName(bundle.getString(Constants.Profile.NAME))
                    .setDescription(bundle.getString(Constants.Profile.DESCRIPTION))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .build();
        }

        @Override
        public Preset[] newArray(int size) {
            return new Preset[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.Profile.PROXIMITY, proximityUUID);
        bundle.putInt(Constants.Profile.INTERVAL, interval);
        bundle.putInt(Constants.Profile.TX_POWER, txPower);
        bundle.putString(Constants.Profile.NAME, name);
        bundle.putString(Constants.Profile.DESCRIPTION, description);
        bundle.putInt(Constants.DATABASE_ID, databaseId);

        dest.writeBundle(bundle);
    }

    /**
     * Creates new Preset instance from JSON content.
     * <p/>
     * The raw content can be obtained via RESTClient.
     *
     * @param jsonObject the json object
     * @return the profile
     */
    public static Preset from(final JSONObject jsonObject) {
        return new Builder()
                .setProximityUUID(JSONUtils.getUUIDOrNull(jsonObject, Constants.Profile.PROXIMITY))
                .setInterval(JSONUtils.getInt(jsonObject, Constants.Profile.INTERVAL, 0))
                .setTxPower(JSONUtils.getInt(jsonObject, Constants.Profile.TX_POWER, 0))
                .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Profile.NAME))
                .setDescription(JSONUtils.getStringOrNull(jsonObject, Constants.Profile.DESCRIPTION))
                .build();
    }

    /**
     * Preset Builder.
     */
    public static final class Builder {

        private int databaseId;
        private UUID proximityUUID;
        private int interval = 0;
        private int txPower = -1;
        private String name;
        private String description;

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the builder
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Builds profile.
         *
         * @return the new Preset instance
         */
        public Preset build() {
            return new Preset(this);
        }

        /**
         * Sets proximityUUID uUID.
         *
         * @param proximity the proximityUUID
         * @return the builder
         */
        public Builder setProximityUUID(UUID proximity) {
            this.proximityUUID = proximity;
            return this;
        }

        /**
         * Sets interval.
         *
         * @param interval the interval
         * @return the builder
         */
        public Builder setInterval(int interval) {
            this.interval = interval;
            return this;
        }

        /**
         * Sets tx power.
         *
         * @param txPower the tx power
         * @return the builder
         */
        public Builder setTxPower(int txPower) {
            this.txPower = txPower;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the builder
         */
        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets description.
         *
         * @param description the description
         * @return the builder
         */
        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }
    }
}
