package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class Namespace extends AbstractModel implements INamespace, Parcelable {
    public static final Creator<Namespace> CREATOR = new Creator<Namespace>() {
        @Override
        public Namespace createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());
            return new Builder()
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setNamespace(bundle.getString(Constants.Namspace.NAMESPACE))
                    .setSecureNamespace(bundle.getString(Constants.Namspace.SECURE_NAMESPACE))
                    .setShuffled(bundle.getBoolean(Constants.Namspace.SHUFFLED))
                    .setShared(bundle.getBoolean(Constants.Namspace.SHARED))
                    .build();
        }

        @Override
        public Namespace[] newArray(int size) {
            return new Namespace[size];
        }
    };

    private final String namespace;
    private final String secureNamespace;
    private final boolean shuffled;
    private final boolean shared;

    private Namespace(Builder builder) {
        super(builder.databaseId);
        this.namespace = builder.namespace;
        this.secureNamespace = builder.secureNamespace;
        this.shuffled = builder.shuffled;
        this.shared = builder.shared;
    }


    @Override
    public String getNamespace() {
        return namespace;
    }

    @Override
    public String getSecureNamespace() {
        return secureNamespace;
    }

    @Override
    public boolean isShuffled() {
        return shuffled;
    }

    @Override
    public boolean isShared() {
        return shared;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(Constants.Namspace.NAMESPACE, namespace);
        bundle.putString(Constants.Namspace.SECURE_NAMESPACE, secureNamespace);
        bundle.putBoolean(Constants.Namspace.SHUFFLED, shuffled);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putBoolean(Constants.Namspace.SHARED, shared);
        dest.writeBundle(bundle);
    }


    public static INamespace from(final JSONObject jsonObject) {
        String namespace = JSONUtils.getStringOrNull(jsonObject, "namespace");
        String secureNamespace = JSONUtils.getStringOrNull(jsonObject, "secureNamespace");
        boolean shuffled = JSONUtils.getBoolean(jsonObject, "shuffled", false);
        boolean shared = JSONUtils.getBoolean(jsonObject, "shared", false);

        return new Namespace.Builder()
                .setShuffled(shuffled)
                .setNamespace(namespace)
                .setSecureNamespace(secureNamespace)
                .setShared(shared)
                .build();
    }

    public static List<INamespace> fromList(final JSONObject jsonObject) {
        try {
            List<INamespace> namespaces = new ArrayList<INamespace>();

            JSONArray jsonArray = jsonObject.getJSONArray("namespaces");

            for (int i = 0; i < jsonArray.length(); i++) {
                namespaces.add(from(jsonArray.getJSONObject(i)));
            }
            return namespaces;
        } catch (JSONException e) {
            throw new IllegalStateException("No namespaces found");
        }
    }

    public static class Builder {
        private int databaseId;
        private String namespace;
        private String secureNamespace;
        private boolean shuffled;
        private boolean shared;

        public Builder setDatabaseId(int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        public Builder setNamespace(String namespace) {
            SDKPreconditions.checkNotNull(namespace, "Namespace cannot be null");
            this.namespace = namespace;
            return this;
        }

        public Builder setSecureNamespace(String secureNamespace) {
            SDKPreconditions.checkNotNull(secureNamespace, "Secure namespace cannot be null");
            this.secureNamespace = secureNamespace;
            return this;
        }

        public Builder setShuffled(boolean shuffled) {
            this.shuffled = shuffled;
            return this;
        }

        public Builder setShared(boolean shared) {
            this.shared = shared;
            return this;
        }

        public Namespace build() {
            return new Namespace(this);
        }
    }
}
