package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;

import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

/**
 * Manager represents person using kontakt.io SDK.
 * The manager may belong to Company.
 * The Manager may have one of three roles: OPERATOR, ADMIN, SUPERUSER.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Manager.Builder}.
 */
public class Manager extends AbstractModel implements Comparable<Manager>, IManager {

    private final UUID id;

    private final UUID supervisorId;

    private final String firstName;

    private final String lastName;

    private final ICompany company;

    private final String email;

    private final String uniqueId;

    private final Role role;

    private final ICounters counters;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<Manager> CREATOR = new Creator<Manager>() {
        public Manager createFromParcel(Parcel input) {
            Bundle bundle = input.readBundle(getClass().getClassLoader());

            return new Builder()
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setSupervisorId((UUID) bundle.getSerializable(Constants.Manager.SUPERVISOR_ID))
                    .setFirstName(bundle.getString(Constants.Manager.FIRST_NAME))
                    .setLastName(bundle.getString(Constants.Manager.LAST_NAME))
                    .setCompany((Company) bundle.getParcelable(Constants.COMPANY))
                    .setEmail(bundle.getString(Constants.Manager.EMAIL))
                    .setUniqueId(bundle.getString(Constants.UNIQUE_ID))
                    .setRole((Role) bundle.getSerializable(Constants.Manager.ROLE))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setCounters((ICounters) bundle.getParcelable(Constants.COUNTERS))
                    .build();
        }

        public Manager[] newArray(int size) {
            return new Manager[size];
        }
    };

    /**
     * Instantiates a new Manager instance.
     *
     * @param builder the builder
     */
    private Manager(final Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.firstName = builder.firstName;
        this.supervisorId = builder.supervisorId;
        this.lastName = builder.lastName;
        this.company = builder.company;
        this.email = builder.email;
        this.uniqueId = builder.uniqueId;
        this.role = builder.role;
        this.counters = builder.counters;

        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .append(firstName)
                .append(supervisorId)
                .append(lastName)
                .append(company)
                .append(email)
                .append(uniqueId)
                .append(role)
                .append(counters)
                .build();
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof Manager)) {
            return false;
        }

        Manager manager = (Manager) o;

        return SDKEqualsBuilder.start()
                .equals(id, manager.id)
                .equals(firstName, manager.firstName)
                .equals(supervisorId, manager.supervisorId)
                .equals(lastName, manager.lastName)
                .equals(company, manager.company)
                .equals(email, manager.email)
                .equals(uniqueId, manager.uniqueId)
                .equals(role, manager.role)
                .equals(counters, manager.counters)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public String toString() {
      return "Manager{" +
          "email='" + email + '\'' +
          ", role=" + role +
          '}';
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public UUID getSupervisorId() {
        return supervisorId;
    }

    @Override
    public String getFirstName() {
        return firstName;
    }

    @Override
    public String getLastName() {
        return lastName;
    }

    @Override
    public ICompany getCompany() {
        return company;
    }

    @Override
    public String getEmail() {
        return email;
    }

    @Override
    public String getUniqueId() {
        return uniqueId;
    }

    @Override
    public Role getRole() {
        return role;
    }

    @Override
    public ICounters getCounters() {
        return counters;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putSerializable(Constants.Manager.SUPERVISOR_ID, supervisorId);
        bundle.putString(Constants.Manager.FIRST_NAME, firstName);
        bundle.putString(Constants.Manager.LAST_NAME, lastName);
        bundle.putParcelable(Constants.COMPANY, company);
        bundle.putString(Constants.Manager.EMAIL, email);
        bundle.putString(Constants.UNIQUE_ID, uniqueId);
        bundle.putSerializable(Constants.Manager.ROLE, role);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putParcelable(Constants.COUNTERS, counters);
        dest.writeBundle(bundle);
    }

    public int compareTo(Manager another) {
        if (another == null || another.uniqueId == null) {
            return 1;
        }

        if (this.uniqueId == null) {
            return -1;
        }

        return (this.uniqueId.compareTo(another.uniqueId));
    }

    /**
     * Instantiates new Manager from JSON content.
     *
     * @param jsonObject the json object with content
     * @return the manager
     */
    public static Manager from(final JSONObject jsonObject) {
        try {
            final UUID id = JSONUtils.getUUIDOrNull(jsonObject, Constants.ID);
            final UUID supervisorId = JSONUtils.getUUIDOrNull(jsonObject, Constants.Manager.SUPERVISOR_ID);
            final String firstName = JSONUtils.getStringOrNull(jsonObject, Constants.Manager.FIRST_NAME);
            final String lastName = JSONUtils.getStringOrNull(jsonObject, Constants.Manager.LAST_NAME);
            final Company company = JSONUtils.hasJSONKey(jsonObject, Constants.COMPANY) ?
                    Company.from(jsonObject.getJSONObject(Constants.COMPANY))
                    : null;
            final String email = JSONUtils.getStringOrNull(jsonObject, Constants.Manager.EMAIL);
            final String uniqueID = JSONUtils.getStringOrNull(jsonObject, Constants.UNIQUE_ID);
            final ICounters counters = JSONUtils.hasJSONKey(jsonObject, Constants.COUNTERS) ?
                    Counters.from(jsonObject.getJSONObject(Constants.COUNTERS)) :
                    null;
            final Role role = JSONUtils.hasJSONKey(jsonObject, Constants.Manager.ROLE) ?
                    Role.valueOf(JSONUtils.getString(jsonObject, Constants.Manager.ROLE, null))
                    :
                    null;
            return new Builder()
                    .setId(id)
                    .setSupervisorId(supervisorId)
                    .setFirstName(firstName)
                    .setLastName(lastName)
                    .setCompany(company)
                    .setEmail(email)
                    .setUniqueId(uniqueID)
                    .setCounters(counters)
                    .setRole(role)
                    .build();
        } catch (JSONException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Creates Set of managers from JSON array.
     *
     * @param jsonArray the json array
     * @return the managers set
     */
    public static Set<Manager> from(final JSONArray jsonArray) {
        final Set<Manager> managerSet = new HashSet<Manager>();

        for (int index = 0, limit = jsonArray.length(); index < limit; index++) {
            try {
                managerSet.add(Manager.from(jsonArray.getJSONObject(index)));
            } catch (JSONException e) {
                throw new IllegalArgumentException(e);
            }
        }

        return managerSet;
    }

    /**
     * Manager Builder.
     */
    public static final class Builder {

        private int databaseId;

        private UUID id;

        private UUID supervisorId;

        private String firstName;

        private String lastName;

        private ICompany company;

        private String email;

        private String uniqueId;

        private ICounters counters;

        private Role role;

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the builder instance
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets manager id.
         *
         * @param id the id
         * @return the builder instance
         */
        public Builder setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets first name.
         *
         * @param firstName the first name
         * @return the builder instance
         */
        public Builder setFirstName(String firstName) {
            this.firstName = firstName;
            return this;
        }

        /**
         * Sets last name.
         *
         * @param lastName the last name
         * @return the builder instance
         */
        public Builder setLastName(String lastName) {
            this.lastName = lastName;
            return this;
        }

        /**
         * Sets company.
         *
         * @param company the company
         * @return the builder instance
         */
        public Builder setCompany(ICompany company) {
            this.company = company;
            return this;
        }

        /**
         * Sets email.
         *
         * @param email the email
         * @return the builder instance
         */
        public Builder setEmail(String email) {
            this.email = email;
            return this;
        }

        /**
         * Sets unique id.
         *
         * @param uniqueId the unique id
         * @return the builder instance
         */
        public Builder setUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }

        /**
         * Sets role.
         *
         * @param role the role
         * @return the builder instance
         */
        public Builder setRole(Role role) {
            this.role = role;
            return this;
        }

        /**
         * Sets supervisor id.
         *
         * @param supervisorId the supervisor id
         * @return the builder instance
         */
        public Builder setSupervisorId(final UUID supervisorId) {
            this.supervisorId = supervisorId;
            return this;
        }

        /**
         * Sets counters.
         *
         * @param counters the counters
         * @return the counters
         */
        public Builder setCounters(ICounters counters) {
            this.counters = counters;
            return this;
        }

        /**
         * Build manager.
         *
         * @return the manager
         */
        public Manager build() {
            return new Manager(this);
        }
    }
}
