package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Represents api response with resolved beacon id (from shuffled) and set of future shuffles
 */
public class IBeaconFutureId implements IFutureId<BeaconId> {

    public static final Creator<IBeaconFutureId> CREATOR = new Creator<IBeaconFutureId>() {
        @Override
        public IBeaconFutureId createFromParcel(Parcel source) {
            return new IBeaconFutureId(source);
        }

        @Override
        public IBeaconFutureId[] newArray(int size) {
            return new IBeaconFutureId[size];
        }
    };


    private String uniqueId;
    private BeaconId queriedBy;
    private BeaconId resolved;
    private List<BeaconId> futureIds;

    public IBeaconFutureId() {

    }

    private IBeaconFutureId(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        queriedBy = bundle.getParcelable(QUERIED_BY);
        resolved = bundle.getParcelable(RESOLVED);
        futureIds = bundle.getParcelableArrayList(FUTURE_SHUFFLES);
        uniqueId = bundle.getString(UNIQUE_ID);
    }

    public BeaconId getQueriedBy() {
        return queriedBy;
    }

    public void setQueriedBy(BeaconId queriedBy) {
        this.queriedBy = queriedBy;
    }

    public BeaconId getResolved() {
        return resolved;
    }

    public void setResolved(BeaconId resolved) {
        this.resolved = resolved;
    }

    public List<BeaconId> getFutureIds() {
        return futureIds;
    }

    public void setFutureIds(List<BeaconId> futureIds) {
        this.futureIds = futureIds;
    }

    public String getUniqueId() {
        return uniqueId;
    }

    public void setUniqueId(String uniqueId) {
        this.uniqueId = uniqueId;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof IBeaconFutureId)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        IBeaconFutureId other = (IBeaconFutureId) o;
        return SDKEqualsBuilder.start()
                .equals(resolved, other.resolved)
                .equals(queriedBy, other.queriedBy)
                .equals(futureIds, other.futureIds)
                .equals(uniqueId, other.uniqueId)
                .result();
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.init()
                .append(resolved)
                .append(queriedBy)
                .append(futureIds)
                .append(uniqueId)
                .build();
    }

    /**
     * Creates IBeaconFutureId from JSONObject
     *
     * @param jsonObject the jsonObject
     * @return IBeaconFutureId
     */
    public static IBeaconFutureId from(JSONObject jsonObject) {
        String uniqueId = JSONUtils.getStringOrNull(jsonObject, UNIQUE_ID);
        String queriedBy = JSONUtils.getStringOrNull(jsonObject, QUERIED_BY);
        BeaconId queriedByBeaconId = BeaconId.fromQueriedBy(queriedBy);
        BeaconId resolvedBeaconId = BeaconId.fromJSONObject(jsonObject);
        List<BeaconId> futureBeaconIds = BeaconId.getIBeaconFutureIds(jsonObject);

        IBeaconFutureId iBeaconShuffles = new IBeaconFutureId();
        iBeaconShuffles.setUniqueId(uniqueId);
        iBeaconShuffles.setQueriedBy(queriedByBeaconId);
        iBeaconShuffles.setResolved(resolvedBeaconId);
        iBeaconShuffles.setFutureIds(futureBeaconIds);

        return iBeaconShuffles;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(UNIQUE_ID, uniqueId);
        bundle.putParcelable(QUERIED_BY, queriedBy);
        bundle.putParcelable(RESOLVED, resolved);
        bundle.putParcelableArrayList(FUTURE_SHUFFLES, (ArrayList<? extends Parcelable>) futureIds);
        dest.writeBundle(bundle);
    }

    @Override
    public int describeContents() {
        return 0;
    }
}
