package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

/**
 * Firmware model provides information concerning the Beacon embedded software
 * that is currently installed on it.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Firmware.Builder}.
 */
public class Firmware extends AbstractModel implements IFirmware {

    private final UUID id;

    private final String name;

    private final String description;

    private final String validVersions;

    private final boolean important;

    private final String url;

    private final DeviceType deviceType;

    private final boolean scheduled;

    private final boolean optional;

    private final ArrayList<String> deviceUniqueIds;


    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<Firmware> CREATOR = new Creator<Firmware>() {
        public Firmware createFromParcel(Parcel parcel) {
            final Bundle bundle = parcel.readBundle(getClass().getClassLoader());
            return new Builder()
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setImportant(bundle.getBoolean(Constants.Firmware.IMPORTANT))
                    .setDescription(bundle.getString(Constants.Firmware.DESCRIPTION))
                    .setValidVersions(bundle.getString(Constants.Firmware.VALID_VERSIONS))
                    .setName(bundle.getString(Constants.Firmware.NAME))
                    .setUrl(bundle.getString(Constants.Firmware.URL))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setDeviceType((DeviceType) bundle.getSerializable(Constants.Firmware.DEVICE_TYPE))
                    .setScheduled(bundle.getBoolean(Constants.Firmware.SCHEDULED))
                    .setOptional(bundle.getBoolean(Constants.Firmware.OPTIONAL))
                    .setDeviceUniqueIds(bundle.getStringArrayList(Constants.Firmware.UNIQUE_ID))
                    .build();
        }

        public Firmware[] newArray(int size) {
            return new Firmware[size];
        }
    };

    /**
     * Instantiates a new Firmware instance.
     *
     * @param builder the builder
     */
    private Firmware(final Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.important = builder.important;
        this.description = builder.description;
        this.validVersions = builder.validVersions;
        this.name = builder.name;
        this.url = builder.url;
        this.deviceType = builder.deviceType;
        this.scheduled = builder.scheduled;
        this.optional = builder.optional;
        List<String> strings = Collections.unmodifiableList(builder.deviceUniqueIds);
        this.deviceUniqueIds = new ArrayList<String>(strings);

        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .build();
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getValidVersions() {
        return validVersions;
    }

    @Override
    public boolean isImportant() {
        return important;
    }

    @Override
    public String getUrl() {
        return url;
    }

    @Override
    public DeviceType getDeviceType() {
        return deviceType;
    }

    @Override
    public boolean isScheduled() {
        return scheduled;
    }

    @Override
    public boolean isOptional() {
        return optional;
    }

    @Override
    public ArrayList<String> getDeviceUniqueIdList() {
        return deviceUniqueIds;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putBoolean(Constants.Firmware.IMPORTANT, important);
        bundle.putString(Constants.Firmware.VALID_VERSIONS, validVersions);
        bundle.putString(Constants.Firmware.NAME, name);
        bundle.putString(Constants.Firmware.DESCRIPTION, description);
        bundle.putString(Constants.Firmware.URL, url);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putSerializable(Constants.Firmware.DEVICE_TYPE, deviceType);
        bundle.putBoolean(Constants.Firmware.SCHEDULED, scheduled);
        bundle.putBoolean(Constants.Firmware.OPTIONAL, optional);
        bundle.putStringArrayList(Constants.Firmware.UNIQUE_ID, deviceUniqueIds);
        dest.writeBundle(bundle);
    }

    @Override
    public boolean equals(Object o) {

        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof Firmware)) {
            return false;
        }

        final Firmware firmware = (Firmware) o;

        return SDKEqualsBuilder.start()
                .equals(id, firmware.id)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    /**
     * Creates new Firmware instance from JSON content and File data.
     *
     * @param jsonObject the json object
     * @return the firmware
     */
    public static Firmware from(final JSONObject jsonObject) {
        String deviceTypeJsonValue = JSONUtils.getString(jsonObject, Constants.Firmware.DEVICE_TYPE, null);
        JSONArray uniqueIdsJsonArray = JSONUtils.getJSONArray(jsonObject, Constants.Firmware.UNIQUE_ID, new JSONArray());
        ArrayList<String> uniqueIds = new ArrayList<String>();
        int uniqueIdSize = uniqueIdsJsonArray.length();
        for (int i = 0; i < uniqueIdSize; i++) {
            uniqueIds.add(JSONUtils.getJSONArrayElement(uniqueIdsJsonArray, i));
        }
        return new Builder()
                .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                .setImportant(JSONUtils.getBoolean(jsonObject, Constants.Firmware.IMPORTANT, false))
                .setValidVersions(JSONUtils.getStringOrNull(jsonObject, Constants.Firmware.VALID_VERSIONS))
                .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Firmware.NAME))
                .setDescription(JSONUtils.getStringOrNull(jsonObject, Constants.Firmware.DESCRIPTION))
                .setUrl(JSONUtils.getString(jsonObject, Constants.Firmware.URL, null))
                .setScheduled(JSONUtils.getBoolean(jsonObject, Constants.Firmware.SCHEDULED, false))
                .setOptional(JSONUtils.getBoolean(jsonObject, Constants.Firmware.OPTIONAL, false))
                .setDeviceType(deviceTypeJsonValue == null ? null : DeviceType.valueOf(deviceTypeJsonValue))
                .setDeviceUniqueIds(uniqueIds)
                .build();
    }

    public static List<IFirmware> fromList(final JSONObject sourceObject) {
        try {
            JSONArray jsonArray = JSONUtils.getJSONArray(sourceObject, Constants.Firmware.FIRMWARES, new JSONArray());
            int jsonArraySize = jsonArray.length();
            List<IFirmware> firmwares = new ArrayList<IFirmware>();
            for (int i = 0; i < jsonArraySize; i++) {
                firmwares.add(from(jsonArray.getJSONObject(i)));
            }
            return firmwares;
        } catch (Exception e) {
            throw new IllegalStateException("No firmwares found");
        }
    }

    /**
     * Firmware Builder.
     */
    public static final class Builder {
        private int databaseId;
        private UUID id;
        private boolean important;
        private String validVersions;
        private String name;
        private String description;
        private String url;
        private DeviceType deviceType;
        private boolean scheduled;
        private boolean optional;
        private ArrayList<String> deviceUniqueIds;

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the builder
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the builder
         */
        public Builder setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets important.
         *
         * @param important the important
         * @return the builder
         */
        public Builder setImportant(boolean important) {
            this.important = important;
            return this;
        }

        /**
         * Sets valid versions.
         *
         * @param validVersions the valid versions
         * @return the builder
         */
        public Builder setValidVersions(String validVersions) {
            this.validVersions = validVersions;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the builder
         */
        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets description.
         *
         * @param description the description
         * @return the builder
         */
        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        /**
         * Sets url.
         *
         * @param url the url
         * @return the builder
         */
        public Builder setUrl(String url) {
            this.url = url;
            return this;
        }

        /**
         * Sets device type or null if not applicable
         *
         * @param deviceType {@link DeviceType}
         * @return the builder
         */
        public Builder setDeviceType(DeviceType deviceType) {
            this.deviceType = deviceType;
            return this;
        }

        /**
         * Sets is firmware scheduled or not
         *
         * @param scheduled boolean value
         * @return the builder
         */
        public Builder setScheduled(boolean scheduled) {
            this.scheduled = scheduled;
            return this;
        }

        /**
         * Sets is firmware update optional or mandatory
         *
         * @param optional boolean value
         * @return the builder
         */
        public Builder setOptional(boolean optional) {
            this.optional = optional;
            return this;
        }

        /**
         * Sets list of device unique ids for which firmware is targeted
         *
         * @param deviceUniqueIds list of device unique ids
         * @return the builder
         */
        public Builder setDeviceUniqueIds(ArrayList<String> deviceUniqueIds) {
            this.deviceUniqueIds = deviceUniqueIds;
            return this;
        }

        /**
         * Build firmware.
         *
         * @return the firmware
         */
        public Firmware build() {
            return new Firmware(this);
        }
    }
}
