package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class EventPacket implements Parcelable {
    private static final String TAG = "EventPacket";

    private static final String TIMESTAMP = "timestamp";
    private static final String EVENTS = "events";

    private final long timestamp;
    private final List<Event> events;

    public static final Creator<EventPacket> CREATOR = new Creator<EventPacket>() {
        @Override
        public EventPacket createFromParcel(Parcel source) {
            return new EventPacket(source);
        }

        @Override
        public EventPacket[] newArray(int size) {
            return new EventPacket[size];
        }
    };

    private EventPacket(Builder builder) {
        this.timestamp = builder.timestamp;
        this.events = builder.events;
    }

    private EventPacket(Parcel source) {
        this.events = new ArrayList<Event>();
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        this.timestamp = bundle.getLong(TIMESTAMP);
        source.readTypedList(events, Event.CREATOR);
    }

    public List<Event> getEvents() {
        return events;
    }

    public long getTimestamp() {
        return timestamp;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putLong(TIMESTAMP, timestamp);
        dest.writeBundle(bundle);
        dest.writeTypedList(events);
    }

    public static class Builder {
        private long timestamp;
        private List<Event> events;

        public Builder setEvents(List<Event> events) {
            SDKPreconditions.checkNotNullOrEmpty(events, "Events list cannot be null or empty");
            this.events = events;
            return this;
        }

        public Builder setTimestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public EventPacket build() {
            return new EventPacket(this);
        }
    }

    public static JSONObject toJSONObject(EventPacket eventPacket) {
        try {
            List<Event> events = eventPacket.getEvents();
            JSONArray jsonEvents = Event.toJSONArray(events);
            JSONObject payload = new JSONObject();
            payload.put(TIMESTAMP, eventPacket.getTimestamp());
            payload.put(EVENTS, jsonEvents);
            return payload;
        } catch (JSONException e) {
            Logger.e(TAG + " Error occurred when try to transform monitoring event packet to JSON", e);
            throw new IllegalArgumentException(e);
        }
    }
}
