package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

public class Event implements Parcelable {

    private static final String TAG = "MonitoringEvent";

    private static final String UNIQUE_ID = "uniqueId";
    private static final String TIMESTAMP = "timestamp";
    private static final String BATTERY_POWER = "batteryPower";
    private static final String BATTERY_LEVEL = "batteryLevel";

    public static final Creator<Event> CREATOR = new Creator<Event>() {
        @Override
        public Event createFromParcel(Parcel source) {
            return new Event(source);
        }

        @Override
        public Event[] newArray(int size) {
            return new Event[size];
        }
    };
    public static final String TYPE = "type";

    private final long timestamp;
    private final int batteryPower;
    private final String uniqueId;
    private final EventType type;

    public Event(EventType type, String uniqueId, int batteryPower, long timestamp) {
        this.type = type;
        this.timestamp = timestamp / 1000;
        this.batteryPower = batteryPower;
        this.uniqueId = uniqueId;
    }

    private Event(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        this.type = EventType.valueOf(bundle.getString(TYPE));
        this.timestamp = bundle.getLong(TIMESTAMP);
        this.uniqueId = bundle.getString(UNIQUE_ID);
        this.batteryPower = bundle.getInt(BATTERY_POWER);
    }

    public static Event of(RemoteBluetoothDevice device, EventType eventType) {
        return new Event(eventType, device.getUniqueId(), device.getBatteryPower(), device.getTimestamp());
    }

    public long getTimestamp() {
        return timestamp;
    }

    public int getBatteryPower() {
        return batteryPower;
    }

    public String getUniqueId() {
        return uniqueId;
    }

    public EventType getType() {
        return type;
    }

    public static JSONObject toJSONObject(Event event) {
        try {
            JSONObject jsonEvent = new JSONObject();
            jsonEvent.put(UNIQUE_ID, event.uniqueId);
            jsonEvent.put(BATTERY_LEVEL, event.batteryPower);
            jsonEvent.put(TIMESTAMP, event.timestamp);
            jsonEvent.put(TYPE, event.type.name());
            return jsonEvent;
        } catch (JSONException e) {
            Logger.e(TAG + " Error occurred when try to transform monitoring event to JSON", e);
            throw new IllegalArgumentException(e);
        }
    }

    public static JSONArray toJSONArray(List<Event> monitoringEvents) {
        JSONArray events = new JSONArray();
        for (Event event : monitoringEvents) {
            events.put(toJSONObject(event));
        }
        return events;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());

        bundle.putString(TYPE, type.name());
        bundle.putString(UNIQUE_ID, uniqueId);
        bundle.putLong(TIMESTAMP, timestamp);
        bundle.putInt(BATTERY_POWER, batteryPower);

        dest.writeBundle(bundle);
    }
}
