package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Representation of resolved shuffled device. Consists of real namespace and instanceId
 */
public class EddystoneUID implements Parcelable {

    public static final Creator<EddystoneUID> CREATOR = new Creator<EddystoneUID>() {
        @Override
        public EddystoneUID createFromParcel(Parcel source) {
            return new EddystoneUID(source);
        }

        @Override
        public EddystoneUID[] newArray(int size) {
            return new EddystoneUID[size];
        }
    };

    private static final String NAMESPACE = "namespace";
    private static final String INSTANCE_ID = "instanceId";
    private static final String FUTURE_SHUFFLES = "futureId";
    private static final String EDDYSTONE = "EDDYSTONE";

    private String namespace;
    private String instanceId;

    Integer hashCode = null;

    /**
     * Creates EddystoneUID with namespace and instanceId
     *
     * @param namespace  the namespace
     * @param instanceId the instanceId
     * @return EddystoneUID
     */
    public static EddystoneUID of(String namespace, String instanceId) {
        return new EddystoneUID(namespace, instanceId);
    }

    /**
     * Creates EddystoneUID from {@link IEddystoneDevice}
     *
     * @param fakeDevice eddystone fake device
     * @return EddystoneUID
     */
    public static EddystoneUID of(IEddystoneDevice fakeDevice) {
        return new EddystoneUID(fakeDevice.getNamespaceId(), fakeDevice.getInstanceId());
    }

    private EddystoneUID(String namespace, String instanceId) {
        this.namespace = namespace;
        this.instanceId = instanceId;
    }

    private EddystoneUID(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        this.namespace = bundle.getString(NAMESPACE);
        this.instanceId = bundle.getString(INSTANCE_ID);
    }

    public String getNamespace() {
        return namespace;
    }

    public String getInstanceId() {
        return instanceId;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    //do not change implementation of toString() method, it is used as http request as well
    @Override
    public String toString() {
        return namespace + ":" + instanceId;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(NAMESPACE, namespace);
        bundle.putString(INSTANCE_ID, instanceId);
        dest.writeBundle(bundle);
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof EddystoneUID)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        EddystoneUID eddystoneUID = (EddystoneUID) o;
        return SDKEqualsBuilder.start()
                .equals(namespace, eddystoneUID.namespace)
                .equals(instanceId, eddystoneUID.instanceId)
                .result();
    }

    @Override
    public int hashCode() {
        if (hashCode == null) {
            hashCode = HashCodeBuilder.init()
                    .append(namespace)
                    .append(instanceId)
                    .build();
        }
        return hashCode;
    }

    /**
     * Creates EddystoneUID from String
     *
     * @param queriedBy the string
     * @return EddystoneUID
     */
    public static EddystoneUID fromQueriedBy(String queriedBy) {

        if (queriedBy == null) {
            throw new NullPointerException("queriedBy is null");
        }

        String[] split = queriedBy.split(":");
        if (split.length != 2) {
            throw new IllegalArgumentException("queriedBy is not valid format");
        }

        String namespace = split[0];
        String instanceId = split[1];

        return new EddystoneUID(namespace, instanceId);
    }

    /**
     * Creates EddystoneUID from JSONObject
     *
     * @param jsonObject the json object
     * @return EddystoneUID
     */
    public static EddystoneUID fromJSONObject(JSONObject jsonObject) {
        String namespace = JSONUtils.getStringOrNull(jsonObject, NAMESPACE);
        if (namespace == null) {
            throw new IllegalStateException("namespace is not present in the response");
        }

        String instanceId = JSONUtils.getStringOrNull(jsonObject, INSTANCE_ID);
        if (instanceId == null) {
            throw new IllegalStateException("instance id is not present in the response");
        }

        return new EddystoneUID(namespace, instanceId);
    }

    /**
     * Creates List of EddystoneUID from JSONObject
     *
     * @param jsonObject the json object
     * @return list of EddystoneUID
     */
    public static List<EddystoneUID> getEddystoneFutureIds(JSONObject jsonObject) {
        List<EddystoneUID> eddystoneUIDs = new ArrayList<EddystoneUID>();
        try {
            JSONObject futureShufflesObject = jsonObject.getJSONObject(FUTURE_SHUFFLES);
            JSONArray eddystoneFutureShuffles = futureShufflesObject.getJSONArray(EDDYSTONE);

            for (int i = 0; i < eddystoneFutureShuffles.length(); i++) {
                JSONObject eddystoneFutureShufflesJSONObject = eddystoneFutureShuffles.getJSONObject(i);
                eddystoneUIDs.add(fromJSONObject(eddystoneFutureShufflesJSONObject));
            }

        } catch (JSONException e) {
            throw new IllegalStateException("futureIds is not present in the response");
        }
        return eddystoneUIDs;
    }
}
