package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Represents api response with resolved eddystone uids (from shuffled) and set of future uids
 */
public class EddystoneFutureUID implements IFutureId<EddystoneUID> {


    public static final Creator<EddystoneFutureUID> CREATOR = new Creator<EddystoneFutureUID>() {
        @Override
        public EddystoneFutureUID createFromParcel(Parcel source) {
            return new EddystoneFutureUID(source);
        }

        @Override
        public EddystoneFutureUID[] newArray(int size) {
            return new EddystoneFutureUID[size];
        }
    };


    private String uniqueId;
    private EddystoneUID queriedBy;
    private EddystoneUID resolved;
    private List<EddystoneUID> futureIds;

    public EddystoneFutureUID() {

    }

    private EddystoneFutureUID(Parcel source) {
        Bundle bundle = source.readBundle(getClass().getClassLoader());
        queriedBy = bundle.getParcelable(QUERIED_BY);
        resolved = bundle.getParcelable(RESOLVED);
        futureIds = bundle.getParcelableArrayList(FUTURE_SHUFFLES);
        uniqueId = bundle.getString(UNIQUE_ID);
    }

    public EddystoneUID getQueriedBy() {
        return queriedBy;
    }

    public void setQueriedBy(EddystoneUID queriedBy) {
        this.queriedBy = queriedBy;
    }

    public EddystoneUID getResolved() {
        return resolved;
    }

    public void setResolved(EddystoneUID resolved) {
        this.resolved = resolved;
    }

    public List<EddystoneUID> getFutureIds() {
        return futureIds;
    }

    public void setFutureIds(List<EddystoneUID> futureUIDs) {
        this.futureIds = futureUIDs;
    }

    public String getUniqueId() {
        return uniqueId;
    }

    public void setUniqueId(String uniqueId) {
        this.uniqueId = uniqueId;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof EddystoneFutureUID)) {
            return false;
        }

        if (o == this) {
            return true;
        }

        EddystoneFutureUID other = (EddystoneFutureUID) o;
        return SDKEqualsBuilder.start()
                .equals(resolved, other.resolved)
                .equals(queriedBy, other.queriedBy)
                .equals(futureIds, other.futureIds)
                .equals(uniqueId, other.uniqueId)
                .result();
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.init()
                .append(resolved)
                .append(queriedBy)
                .append(futureIds)
                .append(uniqueId)
                .build();
    }

    /**
     * Creates EddystoneFutureUID from json object
     *
     * @param jsonObject the json object
     * @return EddystoneFutureUID
     */
    public static EddystoneFutureUID from(JSONObject jsonObject) {
        String uniqueId = JSONUtils.getStringOrNull(jsonObject, UNIQUE_ID);
        String queriedBy = JSONUtils.getStringOrNull(jsonObject, QUERIED_BY);
        EddystoneUID queriedByBeaconId = EddystoneUID.fromQueriedBy(queriedBy);
        EddystoneUID resolvedBeaconId = EddystoneUID.fromJSONObject(jsonObject);
        List<EddystoneUID> futureEddystoneUIDs = EddystoneUID.getEddystoneFutureIds(jsonObject);

        EddystoneFutureUID eddystoneFutureUID = new EddystoneFutureUID();
        eddystoneFutureUID.setUniqueId(uniqueId);
        eddystoneFutureUID.setQueriedBy(queriedByBeaconId);
        eddystoneFutureUID.setResolved(resolvedBeaconId);
        eddystoneFutureUID.setFutureIds(futureEddystoneUIDs);

        return eddystoneFutureUID;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(UNIQUE_ID, uniqueId);
        bundle.putParcelable(QUERIED_BY, queriedBy);
        bundle.putParcelable(RESOLVED, resolved);
        bundle.putParcelableArrayList(FUTURE_SHUFFLES, (ArrayList<? extends Parcelable>) futureIds);
        dest.writeBundle(bundle);
    }

    @Override
    public int describeContents() {
        return 0;
    }

}
