package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * Beacon Credentials model contains password and master password for Beacon.
 * The Credentials can be obtained via API Client.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link DeviceCredentials.Builder}.
 */
public class DeviceCredentials extends AbstractModel implements ICredentials {

    private final String deviceUniqueId;

    private final String password;

    private final String masterPassword;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<DeviceCredentials> CREATOR = new Creator<DeviceCredentials>() {
        @Override
        public DeviceCredentials createFromParcel(Parcel source) {
            final Bundle bundle = source.readBundle(getClass().getClassLoader());
            return new Builder()
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setDeviceUniqueId(bundle.getString(Constants.UNIQUE_ID))
                    .setPassword(bundle.getString("password"))
                    .setMasterPassword(bundle.getString("masterPassword"))
                    .build();
        }

        @Override
        public DeviceCredentials[] newArray(int size) {
            return new DeviceCredentials[size];
        }
    };

    /**
     * Creates new BeaconCredentials instance.
     *
     * @param builder the builder
     */
    private DeviceCredentials(Builder builder) {
        super(builder.databaseId);
        this.deviceUniqueId = builder.deviceUniqueId;
        this.password = builder.password;
        this.masterPassword = builder.masterPassword;

        this.hashCode = HashCodeBuilder.init()
                .append(deviceUniqueId)
                .append(password)
                .append(masterPassword)
                .build();
    }

    @Override
    public String getDeviceUniqueId() {
        return deviceUniqueId;
    }

    @Override
    public String getPassword() {
        return password;
    }

    @Override
    public String getMasterPassword() {
        return masterPassword;
    }


    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }

        if (obj == null || !(obj instanceof DeviceCredentials)) {
            return false;
        }

        final DeviceCredentials credentials = (DeviceCredentials) obj;

        return SDKEqualsBuilder.start()
                .equals(deviceUniqueId, credentials.deviceUniqueId)
                .equals(password, credentials.password)
                .equals(masterPassword, credentials.masterPassword)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putString(Constants.UNIQUE_ID, deviceUniqueId);
        bundle.putString("password", password);
        bundle.putString("masterPassword", masterPassword);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        dest.writeBundle(bundle);
    }

    /**
     * Credentials builder.
     */
    public static class Builder {
        private String deviceUniqueId;

        private String password;

        private String masterPassword;

        private int databaseId;

        /**
         * Sets beacon unique id.
         *
         * @param deviceUniqueId the beacon unique id
         * @return builder
         */
        public Builder setDeviceUniqueId(final String deviceUniqueId) {
            this.deviceUniqueId = deviceUniqueId;
            return this;
        }

        /**
         * Sets password.
         *
         * @param password the password
         * @return builder
         */
        public Builder setPassword(final String password) {
            this.password = password;
            return this;
        }

        /**
         * Sets master password.
         *
         * @param masterPassword the master password
         * @return builder
         */
        public Builder setMasterPassword(final String masterPassword) {
            this.masterPassword = masterPassword;
            return this;
        }

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return builder
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Build beacon credentials.
         *
         * @return the beacon credentials
         */
        public DeviceCredentials build() {
            return new DeviceCredentials(this);
        }
    }
}
