package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * Represents device assigned to the Manager.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Beacon.Builder} or {@link com.kontakt.sdk.android.common.model.CloudBeacon.Builder}.
 */
public abstract class Device extends AbstractModel implements IDevice {

    protected abstract void parcelProperties(final Bundle bundle);

    private final String firmwareVersion;
    private final int interval;
    private final String alias;
    private final int txPower;
    private final String instanceId;
    private final String latitude;
    private final String longitude;
    private final Access access;
    private final IVenue venue;
    private final String url;
    private final String uniqueId;
    private final String namespace;
    private final int actionsCount;
    private final UUID id;
    private final DeviceType deviceType;
    private final UUID managerId;
    private final String name;
    private final Specification specification;
    private final Model model;
    private final List<DeviceProfile> deviceProfiles;
    private final UUID proximityUUID;
    private final int major;
    private final int minor;
    private final String macAddress;

    private final int hashCode;

    protected Device(final Bundle bundle, final Parcel parcel) {
        super(bundle.getInt(Constants.DATABASE_ID));
        this.firmwareVersion = bundle.getString(Constants.Devices.FIRMWARE_VERSION);
        this.interval = bundle.getInt(Constants.Devices.INTERVAL);
        this.alias = bundle.getString(Constants.Devices.ALIAS);
        this.txPower = bundle.getInt(Constants.Devices.TX_POWER);
        this.instanceId = bundle.getString(Constants.Eddystone.INSTANCE_ID);
        this.latitude = bundle.getString(Constants.Devices.LATITUDE);
        this.longitude = bundle.getString(Constants.Devices.LONGITUDE);
        this.access = (Access) bundle.getSerializable(Constants.Devices.ACCESS);
        this.url = bundle.getString(Constants.Eddystone.URL);
        this.uniqueId = bundle.getString(Constants.UNIQUE_ID);
        this.namespace = bundle.getString(Constants.Eddystone.NAMESPACE_ID);
        this.actionsCount = bundle.getInt(Constants.Devices.ACTIONS_COUNT);
        this.id = (UUID) bundle.getSerializable(Constants.ID);
        this.deviceType = (DeviceType) bundle.getSerializable(Constants.Devices.DEVICE_TYPE);
        this.managerId = (UUID) bundle.getSerializable(Constants.Devices.MANAGER_ID);
        this.name = bundle.getString(Constants.Devices.NAME);
        this.specification = (Specification) bundle.getSerializable(Constants.Devices.SPECIFICATION);
        this.model = (Model) bundle.getSerializable(Constants.Devices.MODEL);
        this.proximityUUID = (UUID) bundle.getSerializable(Constants.Devices.PROXIMITY);
        this.major = bundle.getInt(Constants.Devices.MAJOR);
        this.minor = bundle.getInt(Constants.Devices.MINOR);
        this.macAddress = bundle.getString(Constants.Devices.MAC_ADDRESS);

        List<DeviceProfile> deviceProfileList = DeviceProfile.readFromParcel(parcel);
        deviceProfiles = Collections.unmodifiableList(deviceProfileList);

        venue = bundle.getBoolean(PARCELABLE_HAS_VENUE) ?
                Venue.CREATOR.createFromParcel(parcel) :
                null;

        hashCode = calculateHashCode();
    }

    protected Device(Builder builder) {
        super(builder.databaseId);
        this.firmwareVersion = builder.firmwareVersion;
        this.instanceId = builder.instanceId;
        this.alias = builder.alias;
        this.interval = builder.interval;
        this.txPower = builder.txPower;
        this.latitude = builder.latitude;
        this.longitude = builder.longitude;
        this.access = builder.access;
        this.venue = builder.venue;
        this.url = builder.url;
        this.uniqueId = builder.uniqueId;
        this.namespace = builder.namespace;
        this.actionsCount = builder.actionsCount;
        this.id = builder.id;
        this.deviceType = builder.deviceType;
        this.managerId = builder.managerId;
        this.name = builder.name;
        this.specification = builder.specification;
        this.model = builder.model;
        this.deviceProfiles = Collections.unmodifiableList(builder.deviceProfiles);
        this.proximityUUID = builder.proximityUUID;
        this.major = builder.major;
        this.minor = builder.minor;
        this.macAddress = builder.macAddress;
        this.hashCode = calculateHashCode();
    }

    @Override
    // FIXME: why true implementations does not override equals?
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o == null || !(o instanceof Device)) {
            return false;
        }
        Device device = (Device) o;

        return SDKEqualsBuilder.start()
                .equals(uniqueId, device.getUniqueId())
                .equals(firmwareVersion, device.getFirmwareVersion())
                .equals(instanceId, device.getInstanceId())
                .equals(alias, device.getAlias())
                .equals(interval, device.getInterval())
                .equals(txPower, device.getTxPower())
                .equals(latitude, device.getLatitude())
                .equals(longitude, device.getLongitude())
                .equals(access, device.getAccess())
                .equals(venue, device.getVenue())
                .equals(url, device.getUrl())
                .equals(namespace, device.getNamespace())
                .equals(actionsCount, device.getActionsCount())
                .equals(id, device.getId())
                .equals(deviceType, device.getDeviceType())
                .equals(managerId, device.getManagerId())
                .equals(name, device.getName())
                .equals(specification, device.getSpecification())
                .equals(model, device.getModel())
                .equals(deviceProfiles, device.getDeviceProfiles())
                .equals(proximityUUID, device.getProximityUUID())
                .equals(major, device.getMajor())
                .equals(minor, device.getMinor())
                .equals(macAddress, device.getMacAddress())
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public UUID getProximityUUID() {
        return proximityUUID;
    }

    @Override
    public String getMacAddress() {
        return macAddress;
    }

    @Override
    public int getMajor() {
        return major;
    }

    @Override
    public int getMinor() {
        return minor;
    }

    @Override
    public int getInterval() {
        return interval;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getAlias() {
        return alias;
    }

    @Override
    public int getTxPower() {
        return txPower;
    }

    @Override
    public IVenue getVenue() {
        return venue;
    }

    @Override
    public String getUniqueId() {
        return uniqueId;
    }

    @Override
    public DeviceType getDeviceType() {
        return deviceType;
    }

    @Override
    public UUID getManagerId() {
        return managerId;
    }

    @Override
    public int getActionsCount() {
        return actionsCount;
    }

    @Override
    public String getLatitude() {
        return latitude;
    }

    @Override
    public String getLongitude() {
        return longitude;
    }

    @Override
    public Access getAccess() {
        return access;
    }

    @Override
    public Specification getSpecification() {
        return specification;
    }

    @Override public Model getModel() {
        return model;
    }

    @Override
    public List<DeviceProfile> getDeviceProfiles() {
        return deviceProfiles;
    }

    @Override
    public String getFirmwareVersion() {
        return firmwareVersion;
    }

    @Override
    public String getInstanceId() {
        return instanceId;
    }

    @Override
    public String getUrl() {
        return url;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putInt(Constants.DATABASE_ID, getDatabaseId());
        bundle.putString(Constants.Devices.FIRMWARE_VERSION, getFirmwareVersion());
        bundle.putInt(Constants.Devices.INTERVAL, getInterval());
        bundle.putString(Constants.Devices.ALIAS, getAlias());
        bundle.putInt(Constants.Devices.TX_POWER, getTxPower());
        bundle.putString(Constants.Eddystone.INSTANCE_ID, getInstanceId());
        bundle.putString(Constants.Devices.LATITUDE, getLatitude());
        bundle.putString(Constants.Devices.LONGITUDE, getLongitude());
        bundle.putSerializable(Constants.Devices.ACCESS, getAccess());
        bundle.putString(Constants.Eddystone.URL, getUrl());
        bundle.putString(Constants.UNIQUE_ID, getUniqueId());
        bundle.putString(Constants.Eddystone.NAMESPACE_ID, getNamespace());
        bundle.putInt(Constants.Devices.ACTIONS_COUNT, getActionsCount());
        bundle.putSerializable(Constants.ID, getId());
        bundle.putSerializable(Constants.Devices.DEVICE_TYPE, getDeviceType());
        bundle.putSerializable(Constants.Devices.MANAGER_ID, getManagerId());
        bundle.putString(Constants.Devices.NAME, getName());
        bundle.putSerializable(Constants.Devices.SPECIFICATION, getSpecification());
        bundle.putSerializable(Constants.Devices.MODEL, getModel());
        bundle.putInt(Constants.Devices.MAJOR, getMajor());
        bundle.putInt(Constants.Devices.MINOR, getMinor());
        bundle.putString(Constants.Devices.MAC_ADDRESS, getMacAddress());
        bundle.putSerializable(Constants.Devices.PROXIMITY, getProximityUUID());

        final IVenue venue = getVenue();
        final boolean shouldParcelVenue = flags != PARCELABLE_FLAG_EXCLUDE_VENUE && venue != null;
        bundle.putBoolean(PARCELABLE_HAS_VENUE, shouldParcelVenue);

        parcelProperties(bundle);
        dest.writeBundle(bundle);

        List<DeviceProfile> deviceProfiles = getDeviceProfiles();
        DeviceProfile.writeToParcel(dest, deviceProfiles);

        if (shouldParcelVenue) {
            venue.writeToParcel(dest, PARCELABLE_FLAG_EXCLUDE_BEACON);
        }
    }

    protected int calculateHashCode() {
        return HashCodeBuilder.init()
                .append(uniqueId)
                .append(firmwareVersion)
                .append(instanceId)
                .append(alias)
                .append(interval)
                .append(txPower)
                .append(latitude)
                .append(longitude)
                .append(access)
                .append(venue)
                .append(url)
                .append(namespace)
                .append(actionsCount)
                .append(id)
                .append(deviceType)
                .append(managerId)
                .append(name)
                .append(specification)
                .append(model)
                .append(deviceProfiles)
                .append(proximityUUID)
                .append(major)
                .append(minor)
                .append(macAddress)
                .build();
    }

    /**
     * The type Builder.
     *
     * @param <T> the type parameter
     * @param <B> the type parameter
     */
    public abstract static class Builder<T extends Device, B extends Builder<T, B>> {
        int databaseId = -1;
        String firmwareVersion;
        int interval;
        String alias;
        int txPower;
        String instanceId;
        String latitude;
        String longitude;
        Access access;
        IVenue venue;
        String url;
        String uniqueId;
        String namespace;
        int actionsCount;
        UUID id;
        DeviceType deviceType;
        UUID managerId;
        String name;
        Specification specification;
        Model model;
        UUID proximityUUID;
        int major;
        int minor;
        String macAddress;
        List<DeviceProfile> deviceProfiles = new ArrayList<DeviceProfile>();


        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the database id
         */
        public Builder<T, B> setDatabaseId(int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets firmware version.
         *
         * @param firmwareVersion the firmware version
         * @return the firmware version
         */
        public Builder<T, B> setFirmwareVersion(String firmwareVersion) {
            this.firmwareVersion = firmwareVersion;
            return this;
        }

        /**
         * Sets proximity UUID.
         *
         * @param proximity the proximity
         * @return the builder instance
         */
        public Builder<T, B> setProximityUUID(UUID proximity) {
            this.proximityUUID = proximity;
            return this;
        }

        /**
         * Sets interval.
         *
         * @param interval the interval
         * @return the builder instance
         */
        public Builder<T, B> setInterval(int interval) {
            this.interval = interval;
            return this;
        }

        /**
         * Sets alias.
         *
         * @param alias the alias
         * @return the builder instance
         */
        public Builder<T, B> setAlias(String alias) {
            this.alias = alias;
            return this;
        }

        /**
         * Sets tx power.
         *
         * @param txPower the tx power
         * @return the builder instance
         */
        public Builder<T, B> setTxPower(int txPower) {
            this.txPower = txPower;
            return this;
        }

        /**
         * Sets instance id.
         *
         * @param instanceId the instance id
         * @return the instance id
         */
        public Builder<T, B> setInstanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        /**
         * Sets latitude
         *
         * @param latitude latitude value
         * @return the builder instance
         */
        public Builder<T, B> setLatitude(String latitude) {
            this.latitude = latitude;
            return this;
        }

        /**
         * Sets longitude
         *
         * @param longitude longitude value
         * @return the builder instance
         */
        public Builder<T, B> setLongitude(String longitude) {
            this.longitude = longitude;
            return this;
        }

        /**
         * Sets access
         *
         * @param access the {@link Access}
         * @return the builder instance
         */
        public Builder<T, B> setAccess(Access access) {
            this.access = access;
            return this;
        }

        /**
         * Sets venue.
         *
         * @param venue the venue
         * @return the builder instance
         */
        public Builder<T, B> setVenue(IVenue venue) {
            this.venue = venue;
            return this;
        }

        /**
         * Sets url.
         *
         * @param url the url
         * @return the url
         */
        public Builder<T, B> setUrl(String url) {
            this.url = url;
            return this;
        }

        /**
         * Sets unique id.
         *
         * @param uniqueId the unique id
         * @return the builder instance
         */
        public Builder<T, B> setUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }


        /**
         * Sets namespace.
         *
         * @param namespace the namespace
         * @return the namespace
         */
        public Builder<T, B> setNamespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        /**
         * Sets actions count.
         *
         * @param actionsCount the actions count
         * @return the builder instance
         */
        public Builder<T, B> setActionsCount(int actionsCount) {
            this.actionsCount = actionsCount;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the id
         */
        public Builder<T, B> setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets device type.
         *
         * @param deviceType the device type
         * @return the device type
         */
        public Builder<T, B> setDeviceType(DeviceType deviceType) {
            this.deviceType = deviceType;
            return this;
        }

        /**
         * Sets manager id.
         *
         * @param managerId the manager id
         * @return the builder instance
         */
        public Builder<T, B> setManagerId(UUID managerId) {
            this.managerId = managerId;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the builder instance
         */
        public Builder<T, B> setName(String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets specification
         *
         * @param specification the {@link Specification}
         * @return the builder instance
         */
        public Builder<T, B> setSpecification(Specification specification) {
            this.specification = specification;
            return this;
        }

        /**
         * Sets model
         *
         * @param model the {@link Model}
         * @return the builder instance
         */
        public Builder<T, B> setModel(Model model) {
            this.model = model;
            return this;
        }

        /**
         * Sets Device Profiles
         *
         * @param deviceProfiles Collection of DeviceProfile
         * @return the builder instance
         */
        public Builder<T, B> addDeviceProfiles(final Collection<DeviceProfile> deviceProfiles) {
            this.deviceProfiles.addAll(deviceProfiles);
            return this;
        }

        /**
         * Sets major value.
         *
         * @param major the major
         * @return the builder instance
         */
        public Builder<T, B> setMajor(int major) {
            this.major = major;
            return this;
        }

        /**
         * Sets minor value.
         *
         * @param minor the minor
         * @return the builder instance
         */
        public Builder<T, B> setMinor(int minor) {
            this.minor = minor;
            return this;
        }

        /**
         * Sets MAC address value.
         *
         * @param macAddress the MAC address
         * @return the builder instance
         */
        public Builder<T, B> setMacAddress(String macAddress) {
            this.macAddress = macAddress;
            return this;
        }

        /**
         * Build t.
         *
         * @return the t
         */
        public abstract T build();
    }

    /**
     * From device.
     *
     * @param jsonObject the json object
     * @return the Device
     */
    public static Device from(final JSONObject jsonObject) {
        try {
            final DeviceType deviceType = DeviceType.valueOf(jsonObject.getString(Constants.Devices.DEVICE_TYPE));

            switch (deviceType) {
                case BEACON:
                    return Beacon.from(jsonObject);
                case CLOUD_BEACON:
                    return CloudBeacon.from(jsonObject);
                default:
                    throw new IllegalStateException("Unknown device type: " + deviceType.name());
            }
        } catch (Exception e) {
            throw new IllegalStateException("No device type found");
        }
    }

    /**
     * From list of devices
     *
     * @param jsonObject the json object
     * @return List of Device
     */
    public static List<IDevice> fromList(final JSONObject jsonObject) {

        try {
            List<IDevice> devices = new ArrayList<IDevice>();
            JSONArray jsonArray = jsonObject.getJSONArray(Constants.DEVICES);
            for (int i = 0; i < jsonArray.length(); i++) {
                devices.add(from(jsonArray.getJSONObject(i)));
            }
            return devices;
        } catch (Exception e) {
            throw new IllegalStateException("No devices found");
        }
    }
}
