package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;

import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Container for basic information about devices connected with account
 */
public class Counters extends AbstractModel implements ICounters {

    public static final Creator<Counters> CREATOR = new Creator<Counters>() {
        @Override
        public Counters createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle();

            return new Counters.Builder()
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setPrivateDevicesCount(bundle.getInt(Constants.Counters.PRIVATE))
                    .setPublicDevicesCount(bundle.getInt(Constants.Counters.PUBLIC))
                    .setSharedDevicesCount(bundle.getInt(Constants.Counters.SHARED))
                    .setSharedWithMeDevicesCount(bundle.getInt(Constants.Counters.SHARED_WITH_ME))
                    .setPrivateVenuesCount(bundle.getInt(Constants.Counters.PRIVATE + Constants.VENUE))
                    .setPublicVenuesCount(bundle.getInt(Constants.Counters.PUBLIC + Constants.VENUE))
                    .setSharedVenuesCount(bundle.getInt(Constants.Counters.SHARED + Constants.VENUE))
                    .setSharedWithMeVenuesCount(bundle.getInt(Constants.Counters.SHARED_WITH_ME + Constants.VENUE))
                    .build();
        }

        @Override
        public Counters[] newArray(int size) {
            return new Counters[size];
        }
    };

    private final int privateDevicesCount;

    private final int publicDevicesCount;

    private final int sharedDevicesCount;

    private final int sharedWithMeDevicesCount;

    private final int privateVenuesCount;

    private final int publicVenuesCount;

    private final int sharedVenuesCount;

    private final int sharedWithMeVenuesCount;

    private final int hashCode;

    private Counters(Builder builder) {
        super(builder.databaseId);
        this.privateDevicesCount = builder.privateDevicesCount;
        this.publicDevicesCount = builder.publicDevicesCount;
        this.sharedDevicesCount = builder.sharedDevicesCount;
        this.sharedWithMeDevicesCount = builder.sharedWithMeDevicesCount;

        this.privateVenuesCount = builder.privateVenuesCount;
        this.publicVenuesCount = builder.publicVenuesCount;
        this.sharedVenuesCount = builder.sharedVenuesCount;
        this.sharedWithMeVenuesCount = builder.sharedWithMeVenuesCount;

        this.hashCode = HashCodeBuilder.init()
                .append(privateDevicesCount)
                .append(publicDevicesCount)
                .append(sharedDevicesCount)
                .append(sharedWithMeDevicesCount)
                .append(privateVenuesCount)
                .append(publicVenuesCount)
                .append(sharedVenuesCount)
                .append(sharedWithMeVenuesCount)
                .build();
    }

    /**
     * Gets count of private devices
     *
     * @return count
     */
    @Override
    public int getPrivateDevicesCount() {
        return privateDevicesCount;
    }

    /**
     * Gets count of devices shared to public infrastructure
     *
     * @return count
     */
    @Override
    public int getPublicDevicesCount() {
        return publicDevicesCount;
    }

    /**
     * Gets count of devices that are shared
     *
     * @return count
     */
    @Override
    public int getSharedDevicesCount() {
        return sharedDevicesCount;
    }

    /**
     * Gets count of devices that are shared to you
     *
     * @return count
     */
    @Override
    public int getSharedWithMeDevicesCount() {
        return sharedWithMeDevicesCount;
    }

    /**
     * Gets count of private venues
     *
     * @return count
     */
    @Override
    public int getPrivateVenuesCount() {
        return privateVenuesCount;
    }

    /**
     * Gets count of venues that are shared to public infrastructure
     *
     * @return count
     */
    @Override
    public int getPublicVenuesCount() {
        return publicVenuesCount;
    }

    /**
     * Gets count of venues that are shared
     *
     * @return count
     */
    @Override
    public int getSharedVenuesCount() {
        return sharedDevicesCount;
    }

    /**
     * Gets count of venues that are shared to you
     *
     * @return count
     */
    @Override
    public int getSharedWithMeVenuesCount() {
        return sharedWithMeVenuesCount;
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof Counters)) {
            return false;
        }

        Counters counters = (Counters) o;

        return SDKEqualsBuilder.start()
                .equals(this.privateDevicesCount, counters.privateDevicesCount)
                .equals(this.publicDevicesCount, counters.publicDevicesCount)
                .equals(this.sharedDevicesCount, counters.sharedDevicesCount)
                .equals(this.sharedWithMeDevicesCount, counters.sharedWithMeDevicesCount)
                .equals(this.privateVenuesCount, counters.privateVenuesCount)
                .equals(this.publicVenuesCount, counters.publicVenuesCount)
                .equals(this.sharedVenuesCount, counters.sharedVenuesCount)
                .equals(this.sharedWithMeVenuesCount, counters.sharedWithMeVenuesCount)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle();

        bundle.putInt(Constants.DATABASE_ID, databaseId);

        bundle.putInt(Constants.Counters.PRIVATE, privateDevicesCount);
        bundle.putInt(Constants.Counters.PUBLIC, publicDevicesCount);
        bundle.putInt(Constants.Counters.SHARED, sharedDevicesCount);
        bundle.putInt(Constants.Counters.SHARED_WITH_ME, sharedWithMeDevicesCount);

        bundle.putInt(Constants.Counters.PRIVATE + Constants.VENUE, privateVenuesCount);
        bundle.putInt(Constants.Counters.PUBLIC + Constants.VENUE, publicVenuesCount);
        bundle.putInt(Constants.Counters.SHARED + Constants.VENUE, sharedVenuesCount);
        bundle.putInt(Constants.Counters.SHARED_WITH_ME + Constants.VENUE, sharedWithMeVenuesCount);

        dest.writeBundle(bundle);
    }

    /**
     * Creates {@link ICounters} from json object
     *
     * @param jsonObject the json object
     * @return counter
     */
    public static ICounters from(JSONObject jsonObject) {
        try {
            JSONObject devicesJson = jsonObject.getJSONObject(Constants.DEVICES);
            JSONObject venuesJson = jsonObject.getJSONObject(Constants.VENUES);
            return new Counters.Builder()
                    .setPrivateDevicesCount(JSONUtils.getInt(devicesJson, Constants.Counters.PRIVATE, 0))
                    .setPublicDevicesCount(JSONUtils.getInt(devicesJson, Constants.Counters.PUBLIC, 0))
                    .setSharedDevicesCount(JSONUtils.getInt(devicesJson, Constants.Counters.SHARED, 0))
                    .setSharedWithMeDevicesCount(JSONUtils.getInt(devicesJson, Constants.Counters.SHARED_WITH_ME, 0))
                    .setPrivateVenuesCount(JSONUtils.getInt(venuesJson, Constants.Counters.PRIVATE, 0))
                    .setPublicVenuesCount(JSONUtils.getInt(venuesJson, Constants.Counters.PUBLIC, 0))
                    .setSharedDevicesCount(JSONUtils.getInt(venuesJson, Constants.Counters.SHARED, 0))
                    .setSharedWithMeVenuesCount(JSONUtils.getInt(venuesJson, Constants.Counters.SHARED_WITH_ME, 0))
                    .build();
        } catch (JSONException e) {
            Logger.e("Could not create counter", e);
            return null;
        }
    }

    /**
     * Builder class for creating {@link Counters}
     */
    public static final class Builder {

        private int databaseId;

        private int privateDevicesCount;

        private int publicDevicesCount;

        private int sharedDevicesCount;

        private int sharedWithMeDevicesCount;

        private int privateVenuesCount;

        private int publicVenuesCount;

        private int sharedVenuesCount;

        private int sharedWithMeVenuesCount;

        /**
         * Sets database id
         *
         * @param databaseId the database id
         * @return the builder
         */
        public Builder setDatabaseId(int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets private devicec count
         *
         * @param privateDevicesCount the private devices count
         * @return the builder
         */
        public Builder setPrivateDevicesCount(int privateDevicesCount) {
            this.privateDevicesCount = privateDevicesCount;
            return this;
        }

        /**
         * Sets public devices count
         *
         * @param publicDevicesCount the public devices count
         * @return count
         */
        public Builder setPublicDevicesCount(int publicDevicesCount) {
            this.publicDevicesCount = publicDevicesCount;
            return this;
        }

        /**
         * Sets shared devices count
         *
         * @param sharedDevicesCount the shared devices count
         * @return the builder
         */
        public Builder setSharedDevicesCount(int sharedDevicesCount) {
            this.sharedDevicesCount = sharedDevicesCount;
            return this;
        }

        /**
         * Sets shared to user devices count
         *
         * @param sharedWithMeDevicesCount the count
         * @return the builder
         */
        public Builder setSharedWithMeDevicesCount(int sharedWithMeDevicesCount) {
            this.sharedWithMeDevicesCount = sharedWithMeDevicesCount;
            return this;
        }

        /**
         * Sets private venues count
         *
         * @param privateVenuesCount the private venues count
         * @return the builder
         */
        public Builder setPrivateVenuesCount(int privateVenuesCount) {
            this.privateVenuesCount = privateVenuesCount;
            return this;
        }

        /**
         * Sets public venues count
         *
         * @param publicVenuesCount the public venues count
         * @return the builder
         */
        public Builder setPublicVenuesCount(int publicVenuesCount) {
            this.publicVenuesCount = publicVenuesCount;
            return this;
        }

        /**
         * Sets shared venues count
         *
         * @param sharedVenuesCount the shared venues count
         * @return the builder
         */
        public Builder setSharedVenuesCount(int sharedVenuesCount) {
            this.sharedVenuesCount = sharedVenuesCount;
            return this;
        }

        /**
         * Sets shared with user venues count
         *
         * @param sharedWithMeVenuesCount the count
         * @return the builder
         */
        public Builder setSharedWithMeVenuesCount(int sharedWithMeVenuesCount) {
            this.sharedWithMeVenuesCount = sharedWithMeVenuesCount;
            return this;
        }

        /**
         * Creates {@link Counters} object from builder
         *
         * @return counters object
         */
        public Counters build() {
            return new Counters(this);
        }
    }
}
