package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.ContentCategory;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.UUID;

/**
 * Content action contains source url that redirects to content that may be
 * downloaded. The content is described by content category and content type.
 * The Content action can be obtained via REST Client.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link ContentAction.Builder}.
 */
public class ContentAction extends AbstractModel implements IContentAction {

    private final UUID id;

    private final Proximity proximity;

    private final IDevice device;

    private final int contentLength;

    private final String contentType;

    private final String contentUrl;

    private final ContentCategory contentCategory;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<ContentAction> CREATOR = new Creator<ContentAction>() {
        public ContentAction createFromParcel(Parcel input) {
            final Builder builder = new Builder();
            final Bundle bundle = input.readBundle(getClass().getClassLoader());

            if (bundle.getBoolean(PARCELABLE_HAS_BEACON)) {
                builder.setDevice(Beacon.CREATOR.createFromParcel(input));
            }

            return builder
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setProximity((Proximity) bundle.getSerializable(Constants.Action.PROXIMITY))
                    .setContentLength(bundle.getInt(Constants.Action.CONTENT_LENGTH))
                    .setContentType(bundle.getString(Constants.Action.CONTENT_TYPE))
                    .setContentUrl(bundle.getString(Constants.Action.CONTENT))
                    .setContentCategory((ContentCategory) bundle.getSerializable(Constants.Action.CONTENT_CATEGORY))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .build();
        }

        public ContentAction[] newArray(int size) {
            return new ContentAction[size];
        }
    };

    /**
     * Instantiates a new Content action.
     *
     * @param builder the id
     */
    private ContentAction(Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.proximity = builder.proximity;
        this.device = builder.device;
        this.contentLength = builder.contentLength;
        this.contentType = builder.contentType;
        this.contentUrl = builder.contentUrl;
        this.contentCategory = builder.contentCategory;
        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .append(proximity)
                .append(device)
                .append(contentLength)
                .append(contentType)
                .append(contentUrl)
                .append(contentCategory)
                .build();
    }

    @Override
    public boolean equals(Object o) {

        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof ContentAction)) {
            return false;
        }

        ContentAction contentAction = (ContentAction) o;

        return SDKEqualsBuilder.start()
                .equals(id, contentAction.id)
                .equals(proximity, contentAction.proximity)
                .equals(device, contentAction.device)
                .equals(contentLength, contentAction.contentLength)
                .equals(contentType, contentAction.contentType)
                .equals(contentUrl, contentAction.contentUrl)
                .equals(contentCategory, contentAction.contentCategory)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public Proximity getProximity() {
        return proximity;
    }

    @Override
    public ActionType getType() {
        return ActionType.CONTENT;
    }

    @Override
    public IDevice getDevice() {
        return device;
    }

    @Override
    public int getContentLength() {
        return contentLength;
    }

    @Override
    public String getContentType() {
        return contentType;
    }

    @Override
    public String getContentUrl() {
        return contentUrl;
    }

    @Override
    public ContentCategory getContentCategory() {
        return contentCategory;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putSerializable(Constants.Action.PROXIMITY, proximity);
        bundle.putParcelable(Constants.DEVICE, device);
        bundle.putInt(Constants.Action.CONTENT_LENGTH, contentLength);
        bundle.putString(Constants.Action.CONTENT_TYPE, contentType);
        bundle.putString(Constants.Action.CONTENT, contentUrl);
        bundle.putSerializable(Constants.Action.CONTENT_CATEGORY, contentCategory);

        final boolean shouldParcelBeacon = flags != PARCELABLE_FLAG_EXCLUDE_BEACON && device != null;

        bundle.putBoolean(PARCELABLE_HAS_BEACON, shouldParcelBeacon);

        dest.writeBundle(bundle);

        if (shouldParcelBeacon) {
            device.writeToParcel(dest, PARCELABLE_FLAG_EXCLUDE_ACTIONS);
        }

    }

    /**
     * Creates new Content action from JSON content and Content data.
     *
     * @param jsonObject the json object
     * @return the content action
     */
    public static ContentAction from(final JSONObject jsonObject) {
        try {
            return new Builder()
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .setDevice(JSONUtils.hasJSONKey(jsonObject, Constants.DEVICE) ?
                            Beacon.from(jsonObject.getJSONObject(Constants.DEVICE)) : null)
                    .setProximity(Proximity.valueOf(jsonObject.getString(Constants.Action.PROXIMITY)))
                    .setContentLength(JSONUtils.getInt(jsonObject, Constants.Action.CONTENT_LENGTH, 0))
                    .setContentType(JSONUtils.getStringOrNull(jsonObject, Constants.Action.CONTENT_TYPE))
                    .setContentUrl(JSONUtils.getStringOrNull(jsonObject, Constants.Action.CONTENT))
                    .setContentCategory(JSONUtils.hasJSONKey(jsonObject, Constants.Action.CONTENT_CATEGORY) ?
                            ContentCategory.valueOf(jsonObject.getString(Constants.Action.CONTENT_CATEGORY))
                            : ContentCategory.UNKNOWN)
                    .build();
        } catch (JSONException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * The type Builder.
     */
    public static class Builder {
        private int databaseId;
        private UUID id;
        private Proximity proximity;
        private IDevice device;
        private int contentLength;
        private String contentType;
        private String contentUrl;
        private ContentCategory contentCategory;

        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the builder instance
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the builder instance
         */
        public Builder setId(final UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets proximity.
         *
         * @param proximity the proximity
         * @return the builder instance
         */
        public Builder setProximity(final Proximity proximity) {
            this.proximity = proximity;
            return this;
        }

        /**
         * Sets device.
         *
         * @param device the device
         * @return the builder instance
         */
        public Builder setDevice(final IDevice device) {
            this.device = device;
            return this;
        }

        /**
         * Sets content length.
         *
         * @param contentLength the content length
         * @return the builder instance
         */
        public Builder setContentLength(final int contentLength) {
            this.contentLength = contentLength;
            return this;
        }

        /**
         * Sets content type.
         *
         * @param contentType the content type
         * @return the builder instance
         */
        public Builder setContentType(final String contentType) {
            this.contentType = contentType;
            return this;
        }

        /**
         * Sets content url.
         *
         * @param contentUrl the content url
         * @return the builder instance
         */
        public Builder setContentUrl(final String contentUrl) {
            this.contentUrl = contentUrl;
            return this;
        }

        /**
         * Sets content category.
         *
         * @param contentCategory the content category
         * @return the builder instance
         */
        public Builder setContentCategory(final ContentCategory contentCategory) {
            this.contentCategory = contentCategory;
            return this;
        }

        /**
         * Builds Content Action.
         *
         * @return the content action
         */
        public ContentAction build() {
            return new ContentAction(this);
        }
    }
}
