package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

/**
 * Config represents configuration which Beacon device must be updated with.
 * Once the real device becomes updated with parameters matching the Config
 * parameters, the pending Config is removed after sending back the Config
 * by RESTClient with applyConfig() method.
 * <p/>
 * Please, consider fetching and applying Configs for Beacon once you establish
 * connection with Beacon device.
 * <p/>
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Config.Builder}.
 */
public class Config extends AbstractModel implements IConfig, Parcelable {

    private final UUID id;
    private final String uniqueId;
    private final UUID proximityUUID;
    private final int major;
    private final int minor;
    private final int txPower;
    private final int interval;
    private final int hashCode;
    private final String namespace;
    private final String url;
    private final String instanceId;
    private final List<DeviceProfile> deviceProfiles;
    private final Boolean shuffled;
    private final String name;
    private final String devicePassword;

    /**
     * Instantiates a new Config instance.
     *
     * @param builder the builder
     */
    Config(final Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.uniqueId = builder.uniqueId;
        this.proximityUUID = builder.proximityUUID;
        this.major = builder.major;
        this.minor = builder.minor;
        this.txPower = builder.txPower;
        this.interval = builder.interval;
        this.namespace = builder.namespace;
        this.url = builder.url;
        this.instanceId = builder.instanceId;
        this.deviceProfiles = builder.deviceProfiles;
        this.shuffled = builder.shuffled;
        this.name = builder.name;
        this.devicePassword = builder.devicePassword;

        this.hashCode = HashCodeBuilder.init()
                .append(id)
                .append(uniqueId)
                .append(proximityUUID)
                .append(major)
                .append(minor)
                .append(txPower)
                .append(interval)
                .append(namespace)
                .append(url)
                .append(instanceId)
                .append(shuffled)
                .append(name)
                .append(devicePassword)
                .build();
    }

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     * <li>
     * There may be situations in which parent object may contain a member holding reference to the parent object
     * in its member. Once such object is parceled, every child will be recreated with its member set to null.
     * This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     * </li>
     * <li>
     * Every model that contains {@link FileData} as its member will be recreated
     * with this member set to null. Please organise the access to the data in different way.
     * </li>
     * </ul>
     * <p/>
     * <p/>
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Creator<Config> CREATOR = new Creator<Config>() {
        @Override
        public Config createFromParcel(Parcel source) {
            final Bundle bundle = source.readBundle(getClass().getClassLoader());
            List<DeviceProfile> deviceProfileList = DeviceProfile.readFromParcel(source);

            return new Builder()
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setDeviceUniqueId(bundle.getString(Constants.Config.UNIQUE_ID))
                    .setProximityUUID((UUID) bundle.getSerializable(Constants.Config.PROXIMITY))
                    .setMajor(bundle.getInt(Constants.Config.MAJOR))
                    .setMinor(bundle.getInt(Constants.Config.MINOR))
                    .setTxPower(bundle.getInt(Constants.Config.TX_POWER))
                    .setInterval(bundle.getInt(Constants.Config.INTERVAL))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .setNamespace(bundle.getString(Constants.Eddystone.NAMESPACE))
                    .setInstanceId(bundle.getString(Constants.Eddystone.INSTANCE_ID))
                    .setUrl(bundle.getString(Constants.Eddystone.URL))
                    .setDeviceProfiles(deviceProfileList)
                    .setShuffled(bundle.getBoolean(Constants.Config.SHUFFLED))
                    .setName(bundle.getString(Constants.Devices.NAME))
                    .setPassword(bundle.getString(Constants.Devices.PASSWORD))
                    .build();
        }

        @Override
        public Config[] newArray(int size) {
            return new Config[size];
        }
    };

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public String getDeviceUniqueId() {
        return uniqueId;
    }

    @Override
    public UUID getProximityUUID() {
        return proximityUUID;
    }

    @Override
    public int getMajor() {
        return major;
    }

    @Override
    public int getMinor() {
        return minor;
    }

    @Override
    public int getTxPower() {
        return txPower;
    }

    @Override
    public int getInterval() {
        return interval;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    @Override
    public String getInstanceId() {
        return instanceId;
    }

    @Override
    public String getUrl() {
        return url;
    }

    @Override
    public List<DeviceProfile> getDeviceProfiles() {
        return deviceProfiles;
    }

    @Override
    public Boolean isShuffled() {
        return shuffled;
    }

    @Override
    public String getDeviceName() {
        return name;
    }


    @Override
    public String getDevicePassword() {
        return devicePassword;
    }

    @Override
    public boolean equals(Object object) {

        if (object == this) {
            return true;
        }

        if (object == null || !(object instanceof Config)) {
            return false;
        }

        final Config config = (Config) object;

        return SDKEqualsBuilder.start()
                .equals(id, config.id)
                .equals(uniqueId, config.uniqueId)
                .equals(proximityUUID, config.proximityUUID)
                .equals(major, config.major)
                .equals(minor, config.minor)
                .equals(txPower, config.txPower)
                .equals(interval, config.interval)
                .equals(namespace, config.namespace)
                .equals(url, config.url)
                .equals(instanceId, config.instanceId)
                .equals(name, config.name)
                .equals(devicePassword, config.devicePassword)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putString(Constants.Config.UNIQUE_ID, uniqueId);
        bundle.putSerializable(Constants.Config.PROXIMITY, proximityUUID);
        bundle.putInt(Constants.Config.MAJOR, major);
        bundle.putInt(Constants.Config.MINOR, minor);
        bundle.putInt(Constants.Config.TX_POWER, txPower);
        bundle.putInt(Constants.Config.INTERVAL, interval);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        bundle.putString(Constants.Eddystone.NAMESPACE, getNamespace());
        bundle.putString(Constants.Eddystone.URL, getUrl());
        bundle.putString(Constants.Eddystone.INSTANCE_ID, getInstanceId());
        if (shuffled != null) {
            bundle.putBoolean(Constants.Config.SHUFFLED, shuffled);
        }
        bundle.putString(Constants.Devices.NAME, name);
        bundle.putString(Constants.Devices.PASSWORD, devicePassword);
        dest.writeBundle(bundle);

        List<DeviceProfile> deviceProfiles = getDeviceProfiles();
        DeviceProfile.writeToParcel(dest, deviceProfiles);
    }

    /**
     * Create Config instance from JSON content.
     *
     * @param jsonObject the json object
     * @return the config
     */
    public static Config from(final JSONObject jsonObject) {
        try {
            String hexedUrl = JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.URL);

            Builder builder = new Builder()
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .setDeviceUniqueId(JSONUtils.getStringOrNull(jsonObject, Constants.Config.UNIQUE_ID))
                    .setProximityUUID(JSONUtils.getUUIDOrNull(jsonObject, Constants.Config.PROXIMITY))
                    .setMajor(JSONUtils.getInt(jsonObject, Constants.Config.MAJOR, -1))
                    .setMinor(JSONUtils.getInt(jsonObject, Constants.Config.MINOR, -1))
                    .setTxPower(JSONUtils.getInt(jsonObject, Constants.Config.TX_POWER, -1))
                    .setInterval(JSONUtils.getInt(jsonObject, Constants.Config.INTERVAL, -1))
                    .setNamespace(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.NAMESPACE))
                    .setUrl(hexedUrl != null ? EddystoneUtils.fromHexedUrlToUrl(hexedUrl) : null)
                    .setInstanceId(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.INSTANCE_ID))
                    .setShuffled(JSONUtils.getBooleanBoxed(jsonObject, Constants.Config.SHUFFLED))
                    .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.NAME))
                    .setPassword(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.PASSWORD));

            JSONArray profiles = jsonObject.optJSONArray(Constants.Devices.PROFILES);
            if (profiles != null) {
                List<DeviceProfile> deviceProfileList = new ArrayList<DeviceProfile>();
                int profilesLength = profiles.length();
                for (int i = 0; i < profilesLength; i++) {
                    deviceProfileList.add(DeviceProfile.valueOf((String) profiles.get(i)));
                }
                builder.setDeviceProfiles(deviceProfileList);
            }
            return builder.build();

        } catch (JSONException ignored) {

        }
        return null;

    }

    /**
     * Config Builder.
     */
    public static class Builder {
        int databaseId;
        private UUID id;
        private String uniqueId;
        private UUID proximityUUID;
        private int major = 0;
        private int minor = 0;
        private int txPower = -1;

        private int interval = 0;
        private String namespace;
        private String url;
        private String instanceId;
        private List<DeviceProfile> deviceProfiles = new ArrayList<DeviceProfile>();
        private Boolean shuffled;
        private String name;
        private String devicePassword;

        /**
         * Builds config.
         *
         * @return the config
         */
        public Config build() {
            return new Config(this);
        }


        /**
         * Sets database id.
         *
         * @param databaseId the database id
         * @return the database id
         */
        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets beacon unique id.
         *
         * @param uniqueId the unique id
         * @return the builder instance
         */
        public Builder setDeviceUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return the builder instance
         */
        public Builder setId(final UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets proximity uUID.
         *
         * @param proximity the proximity
         * @return the builder instance
         */
        public Builder setProximityUUID(UUID proximity) {
            this.proximityUUID = proximity;
            return this;
        }

        /**
         * Sets major.
         *
         * @param major the major
         * @return the builder instance
         */
        public Builder setMajor(int major) {
            this.major = major;
            return this;
        }

        /**
         * Sets minor.
         *
         * @param minor the minor
         * @return the builder instance
         */
        public Builder setMinor(int minor) {
            this.minor = minor;
            return this;
        }

        /**
         * Sets tx power.
         *
         * @param txPower the tx power
         * @return the builder instance
         */
        public Builder setTxPower(int txPower) {
            this.txPower = txPower;
            return this;
        }

        /**
         * Sets interval.
         *
         * @param interval the interval
         * @return the builder instance
         */
        public Builder setInterval(int interval) {
            this.interval = interval;
            return this;
        }


        /**
         * Sets namespace.
         *
         * @param namespace the namespace
         * @return the namespace
         */
        public Builder setNamespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        /**
         * Sets url.
         *
         * @param url the url
         * @return the url
         */
        public Builder setUrl(String url) {
            this.url = url;
            return this;
        }

        /**
         * Sets instance id.
         *
         * @param instanceId the instance id
         * @return the instance id
         */
        public Builder setInstanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public Builder setDeviceProfiles(Collection<DeviceProfile> deviceProfiles) {
            this.deviceProfiles.addAll(deviceProfiles);
            return this;
        }

        public Builder setShuffled(Boolean shuffled) {
            this.shuffled = shuffled;
            return this;
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setPassword(String devicePassword) {
            this.devicePassword = devicePassword;
            return this;
        }
    }
}
