package com.kontakt.sdk.android.common.model;


import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONObject;

import java.util.UUID;

/**
 * Company model describes real Company registered in kontakt.io database.
 *
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link Company.Builder}.
 */
public class Company extends AbstractModel implements ICompany {

    private final UUID id;

    private final String countryCode;

    private final String name;

    private final String uniqueId;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     *     <li>
     *         There may be situations in which parent object may contain a member holding reference to the parent object
     *         in its member. Once such object is parceled, every child will be recreated with its member set to null.
     *         This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     *     </li>
     *     <li>
     *         Every model that contains {@link FileData} as its member will be recreated
     *         with this member set to null. Please organise the access to the data in different way.
     *     </li>
     * </ul>
     *
     *
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     *
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     *
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Parcelable.Creator<Company> CREATOR = new Parcelable.Creator<Company>() {
        public Company createFromParcel(Parcel input) {
            final Bundle bundle = input.readBundle(getClass().getClassLoader());
            return new Builder()
                    .setUniqueId(bundle.getString(Constants.UNIQUE_ID))
                    .setId((UUID) bundle.getSerializable(Constants.ID))
                    .setCountryCode(bundle.getString(Constants.Company.COUNTRY_CODE))
                    .setName(bundle.getString(Constants.Company.NAME))
                    .setDatabaseId(bundle.getInt(Constants.DATABASE_ID))
                    .build();
        }

        public Company[] newArray(int size) {
            return new Company[size];
        }
    };

    /**
     * Instantiates a new Company.
     *
     * @param builder the builder
     */
    private Company(final Builder builder) {
        super(builder.databaseId);
        this.id = builder.id;
        this.countryCode = builder.countryCode;
        this.name = builder.name;
        this.uniqueId = builder.uniqueId;

        this.hashCode = HashCodeBuilder.init()
                                        .append(id)
                                        .append(countryCode)
                                        .append(name)
                                        .append(uniqueId)
                                        .build();
    }

    @Override
    public boolean equals(Object obj) {
        if(obj == this) {
            return true;
        }

        if(obj == null || !(obj instanceof Company)) {
            return false;
        }

        final Company company = (Company) obj;

        return SDKEqualsBuilder.start()
                .equals(id, company.id)
                .equals(countryCode, company.countryCode)
                .equals(name, company.name)
                .equals(uniqueId, company.uniqueId)
                .result();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public UUID getId() {
        return id;
    }

    @Override
    public String getCountryCode() {
        return countryCode;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getUniqueId() {
        return uniqueId;
    }

    /**
     * Creates new Company instance from JSON content.
     *
     * @param jsonObject the json object
     * @return the company
     */
    public static Company from(final JSONObject jsonObject) {
        return new Builder()
                .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                .setCountryCode(JSONUtils.getStringOrNull(jsonObject, Constants.Company.COUNTRY_CODE))
                .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Company.NAME))
                .setUniqueId(JSONUtils.getStringOrNull(jsonObject, Constants.UNIQUE_ID))
                .build();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        final Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putSerializable(Constants.ID, id);
        bundle.putString(Constants.Company.COUNTRY_CODE, countryCode);
        bundle.putString(Constants.Company.NAME, name);
        bundle.putString(Constants.UNIQUE_ID, uniqueId);
        bundle.putInt(Constants.DATABASE_ID, databaseId);
        dest.writeBundle(bundle);
    }

    /**
     * Company Builder.
     */
    public static class Builder {
        private int databaseId;
        private UUID id;
        private String countryCode;
        private String name;
        private String uniqueId;

        public Builder setDatabaseId(final int databaseId) {
            this.databaseId = databaseId;
            return this;
        }

        /**
         * Sets Company Id.
         *
         * @param id the id
         * @return Builder instance
         */
        public Builder setId(UUID id) {
            this.id = id;
            return this;
        }

        /**
         * Sets country code.
         *
         * @param countryCode the country code
         * @return Builder instance
         */
        public Builder setCountryCode(String countryCode) {
            this.countryCode = countryCode;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return Builder instance
         */
        public Builder setName(String name) {
            this.name = name;
            return this;

        }

        /**
         * Sets unique id.
         *
         * @param uniqueId the unique id
         * @return Builder instance
         */
        public Builder setUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }

        /**
         * Builds company instance.
         *
         * @return the company
         */
        public Company build() {
            return new Company(this);
        }
    }
}
