package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;
import java.util.UUID;

/**
 * Cloud config represents the configuration that can be applied
 * to Cloud Beacon device.
 *
 * By convention this model is fully immutable.
 * To create new instance of the model, please use the {@link CloudConfig.Builder}.
 */
public final class CloudConfig extends AbstractModel implements ICloudConfig {

    private final Config config;

    private final String defaultSSIDName;
    private final String defaultSSIDKey;
    private final String defaultSSIDAuth;
    private final String defaultSSIDCrypt;
    private final String password;
    private final String name;
    private final ICloudBeacon.WorkingMode workingMode;
    private final int wifiScanInterval;
    private final int dataSendInterval;
    private final int bleScanInterval;
    private final int bleScanDuration;

    private final int hashCode;

    /**
     * Parcelable CREATOR constant.
     * This model may be put into Bundle once you decide to save its state.
     * However, please be aware of some limitations.
     * <ul>
     *     <li>
     *         There may be situations in which parent object may contain a member holding reference to the parent object
     *         in its member. Once such object is parceled, every child will be recreated with its member set to null.
     *         This limitation prevents from infinite parceling recursion causing {@link StackOverflowError}
     *     </li>
     *     <li>
     *         Every model that contains {@link FileData} as its member will be recreated
     *         with this member set to null. Please organise the access to the data in different way.
     *     </li>
     * </ul>
     *
     *
     * For more information concerning parceling see attached links.
     *
     * @see <a href="http://developer.android.com/reference/android/os/Parcelable.html" target="_blank">Android SDK documentation - Parcelable</a>
     *
     * @see <a href="http://developer.android.com/reference/android/app/Activity.html#onSaveInstanceState(android.os.Bundle)" target="_blank">Android SDK documentation - Activity.onSaveInstanceState(android.os.Bundle) method</a>
     *
     * @see <a href="http://developer.android.com/reference/android/os/Bundle.html" target="_blank">Android SDK documentation - Bundle</a>
     */
    public static final Parcelable.Creator<CloudConfig> CREATOR = new Parcelable.Creator<CloudConfig>() {
        @Override
        public CloudConfig createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());

            Config config = bundle.getParcelable("config");
            return new Builder()
                    .setDatabaseId(config.getDatabaseId())
                    .setId(config.getId())
                    .setBleScanDuration(bundle.getInt(Constants.Devices.BLE_SCAN_DURATION))
                    .setBleScanInterval(bundle.getInt(Constants.Devices.BLE_SCAN_INTERVAL))
                    .setDataSendInterval(bundle.getInt(Constants.Devices.DATA_SEND_INTERVAL))
                    .setDefaultSSIDAuth(bundle.getString(Constants.Devices.DEFAULT_SSID_AUTH))
                    .setDefaultSSIDCrypt(bundle.getString(Constants.Devices.DEFAULT_SSID_CRYPT))
                    .setDefaultSSIDKey(bundle.getString("defaultSSIDKey"))
                    .setDefaultSSIDName(bundle.getString(Constants.Devices.DEFAULT_SSID_NAME))
                    .setWifiScanInterval(bundle.getInt(Constants.Devices.WIFI_SCAN_INTERVAL))
                    .setProximityUUID(config.getProximityUUID())
                    .setDeviceUniqueId(config.getDeviceUniqueId())
                    .setInterval(config.getInterval())
                    .setMajor(config.getMajor())
                    .setMinor(config.getMinor())
                    .setTxPower(config.getTxPower())
                    .setDataSendInterval(bundle.getInt(Constants.Devices.DATA_SEND_INTERVAL))
                    .setPassword(bundle.getString(Constants.Devices.PASSWORD))
                    .setName(bundle.getString(Constants.Devices.NAME))
                    .setWorkingMode((ICloudBeacon.WorkingMode) bundle.getSerializable(Constants.Devices.WORKING_MODE))
                    .build();
        }

        @Override
        public CloudConfig[] newArray(int size) {
            return new CloudConfig[size];
        }
    };

    private CloudConfig(Builder builder) {
        super(builder.configBuilder.databaseId);

        config = new Config(builder.configBuilder);

        this.defaultSSIDName = builder.defaultSSIDName;
        this.defaultSSIDKey = builder.defaultSSIDKey;
        this.defaultSSIDAuth = builder.defaultSSIDAuth;
        this.defaultSSIDCrypt = builder.defaultSSIDCrypt;
        this.workingMode = builder.workingMode;
        this.wifiScanInterval = builder.wifiScanInterval;
        this.dataSendInterval = builder.dataSendInterval;
        this.bleScanInterval = builder.bleScanInterval;
        this.bleScanDuration = builder.bleScanDuration;
        this.password = builder.password;
        this.name = builder.name;

        this.hashCode = HashCodeBuilder.init()
                .append(config.hashCode())
                .append(defaultSSIDName)
                .append(defaultSSIDKey)
                .append(defaultSSIDAuth)
                .append(defaultSSIDCrypt)
                .append(workingMode)
                .append(wifiScanInterval)
                .append(dataSendInterval)
                .append(bleScanInterval)
                .append(bleScanDuration)
                .append(password)
                .append(name)
                .build();
    }

    @Override
    public String getDefaultSSIDName() {
        return defaultSSIDName;
    }

    @Override
    public String getDefaultSSIDKey() {
        return defaultSSIDKey;
    }

    @Override
    public String getDefaultSSIDAuth() {
        return defaultSSIDAuth;
    }

    @Override
    public String getDefaultSSIDCrypt() {
        return defaultSSIDCrypt;
    }

    @Override
    public String getPassword() {
        return password;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public ICloudBeacon.WorkingMode getWorkingMode() {
        return workingMode;
    }

    @Override
    public int getWifiScanInterval() {
        return wifiScanInterval;
    }

    @Override
    public int getDataSendInterval() {
        return dataSendInterval;
    }

    @Override
    public int getBleScanInterval() {
        return bleScanInterval;
    }

    @Override
    public int getBleScanDuration() {
        return bleScanDuration;
    }

    @Override
    public UUID getId() {
        return config.getId();
    }

    @Override
    public String getDeviceUniqueId() {
        return config.getDeviceUniqueId();
    }

    @Override
    public UUID getProximityUUID() {
        return config.getProximityUUID();
    }

    @Override
    public int getMajor() {
        return config.getMajor();
    }

    @Override
    public int getMinor() {
        return config.getMinor();
    }

    @Override
    public int getTxPower() {
        return config.getTxPower();
    }

    @Override
    public int getInterval() {
        return config.getInterval();
    }

    @Override
    public String getNamespace() {
        return null;
    }

    @Override
    public String getUrl() {
        return null;
    }

    @Override
    public String getInstanceId() {
        return null;
    }

    @Override
    public List<DeviceProfile> getDeviceProfiles() {
        return null;
    }

    @Override
    public Boolean isShuffled() {
        return false;
    }

    @Override
    public String getDeviceName() {
        return null;
    }

    @Override
    public String getDevicePassword() {
        return null;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        Bundle bundle = new Bundle(getClass().getClassLoader());
        bundle.putParcelable("config", config);
        bundle.putString(Constants.Devices.DEFAULT_SSID_NAME, defaultSSIDName);
        bundle.putString("defaultSSIDKey", defaultSSIDKey);
        bundle.putString(Constants.Devices.DEFAULT_SSID_AUTH, defaultSSIDAuth);
        bundle.putString(Constants.Devices.DEFAULT_SSID_CRYPT, defaultSSIDCrypt);
        bundle.putString(Constants.Devices.PASSWORD, password);
        bundle.putString(Constants.Devices.NAME, name);
        bundle.putSerializable(Constants.Devices.WORKING_MODE, workingMode);
        bundle.putInt(Constants.Devices.WIFI_SCAN_INTERVAL, wifiScanInterval);
        bundle.putInt(Constants.Devices.DATA_SEND_INTERVAL, dataSendInterval);
        bundle.putInt(Constants.Devices.BLE_SCAN_DURATION, bleScanDuration);
        bundle.putInt(Constants.Devices.BLE_SCAN_INTERVAL, bleScanInterval);

        dest.writeBundle(bundle);
    }

    @Override
    public boolean equals(Object obj) {
        if(obj == this) {
            return true;
        }

        if(obj == null || !(obj instanceof CloudConfig)) {
            return false;
        }

        CloudConfig cloudConfig = (CloudConfig) obj;

        return SDKEqualsBuilder.start()
                .equals(cloudConfig.config, cloudConfig.config)
                .equals(defaultSSIDName, cloudConfig.defaultSSIDName)
                .equals(defaultSSIDKey, cloudConfig.defaultSSIDKey)
                .equals(defaultSSIDAuth, cloudConfig.defaultSSIDAuth)
                .equals(defaultSSIDCrypt, cloudConfig.defaultSSIDCrypt)
                .equals(workingMode, cloudConfig.workingMode)
                .equals(wifiScanInterval, cloudConfig.wifiScanInterval)
                .equals(dataSendInterval, cloudConfig.dataSendInterval)
                .equals(bleScanInterval, cloudConfig.bleScanInterval)
                .equals(bleScanDuration, cloudConfig.bleScanDuration)
                .equals(password, cloudConfig.password)
                .equals(name, cloudConfig.name)
                .result();
    }

    /**
     * From cloud config.
     *
     * @param jsonObject the json object
     * @return the cloud config
     */
    public static CloudConfig from(final JSONObject jsonObject) {
        try {
            return new Builder()
                    .setWifiScanInterval(JSONUtils.getInt(jsonObject, Constants.Devices.WIFI_SCAN_INTERVAL, -1))
                    .setBleScanDuration(JSONUtils.getInt(jsonObject, Constants.Devices.BLE_SCAN_DURATION, -1))
                    .setInterval(JSONUtils.getInt(jsonObject, Constants.Devices.INTERVAL, -1))
                    .setWorkingMode(JSONUtils.hasJSONKey(jsonObject, Constants.Devices.WORKING_MODE) ?
                            ICloudBeacon.WorkingMode.valueOf(jsonObject.getString(Constants.Devices.WORKING_MODE)) :
                            null)
                    .setDefaultSSIDAuth(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_AUTH))
                    .setMinor(JSONUtils.getInt(jsonObject, Constants.Devices.MINOR, -1))
                    .setTxPower(JSONUtils.getInt(jsonObject, Constants.Devices.TX_POWER, -1))
                    .setPassword(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.PASSWORD))
                    .setDeviceUniqueId(JSONUtils.getStringOrNull(jsonObject, Constants.UNIQUE_ID))
                    .setDataSendInterval(JSONUtils.getInt(jsonObject, Constants.Devices.DATA_SEND_INTERVAL, -1))
                    .setProximityUUID(UUID.fromString(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.PROXIMITY)))
                    .setDefaultSSIDCrypt(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_CRYPT))
                    .setBleScanInterval(JSONUtils.getInt(jsonObject, Constants.Devices.BLE_SCAN_INTERVAL, -1))
                    .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.NAME))
                    .setDefaultSSIDName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_NAME))
                    .setMajor(JSONUtils.getInt(jsonObject, Constants.Devices.MAJOR, -1))
                    .build();
        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * The type Builder.
     */
    public static final class Builder {
        /**
         * The Config builder.
         */
        final Config.Builder configBuilder = new Config.Builder();
        private String defaultSSIDName;
        private String defaultSSIDKey;
        private String defaultSSIDAuth;
        private String defaultSSIDCrypt;
        private ICloudBeacon.WorkingMode workingMode;
        private int wifiScanInterval;
        private int dataSendInterval;
        private int bleScanInterval;
        private int bleScanDuration;
        private String password;
        private String name;

        public Builder setDatabaseId(final int databaseId) {
            configBuilder.setDatabaseId(databaseId);
            return this;
        }

        /**
         * Sets beacon unique id.
         *
         * @param uniqueId the beacon unique id
         * @return Builder instance
         */
        public Builder setDeviceUniqueId(String uniqueId) {
            configBuilder.setDeviceUniqueId(uniqueId);
            return this;
        }

        /**
         * Sets id.
         *
         * @param id the id
         * @return Builder instance
         */
        public Builder setId(final UUID id) {
            configBuilder.setId(id);
            return this;
        }

        /**
         * Sets proximity UUID.
         *
         * @param proximity the proximity
         * @return Builder instance
         */
        public Builder setProximityUUID(UUID proximity) {
            configBuilder.setProximityUUID(proximity);
            return this;
        }

        /**
         * Sets major.
         *
         * @param major the major
         * @return Builder instance
         */
        public Builder setMajor(int major) {
            configBuilder.setMajor(major);
            return this;
        }

        /**
         * Sets minor.
         *
         * @param minor the minor
         * @return Builder instance
         */
        public Builder setMinor(int minor) {
            configBuilder.setMinor(minor);
            return this;
        }

        /**
         * Sets tx power.
         *
         * @param txPower the tx power
         * @return Builder instance
         */
        public Builder setTxPower(int txPower) {
            configBuilder.setTxPower(txPower);
            return this;
        }

        /**
         * Sets interval.
         *
         * @param interval the interval
         * @return Builder instance
         */
        public Builder setInterval(int interval) {
            configBuilder.setInterval(interval);
            return this;
        }

        /**
         * Sets default SSID name.
         *
         * @param defaultSSIDName the default sSID name
         * @return the default sSID name
         */
        public Builder setDefaultSSIDName(String defaultSSIDName) {
            this.defaultSSIDName = defaultSSIDName;
            return this;
        }

        /**
         * Sets default SSID key.
         *
         * @param defaultSSIDKey the default sSID key
         * @return the default sSID key
         */
        public Builder setDefaultSSIDKey(String defaultSSIDKey) {
            this.defaultSSIDKey = defaultSSIDKey;
            return this;
        }

        /**
         * Sets default SSID Auth.
         *
         * @param defaultSSIDAuth the default sSID auth
         * @return the default sSID auth
         */
        public Builder setDefaultSSIDAuth(String defaultSSIDAuth) {
            this.defaultSSIDAuth = defaultSSIDAuth;
            return this;
        }

        /**
         * Sets default SSID crypt.
         *
         * @param defaultSSIDCrypt the default sSID crypt
         * @return the default sSID crypt
         */
        public Builder setDefaultSSIDCrypt(String defaultSSIDCrypt) {
            this.defaultSSIDCrypt = defaultSSIDCrypt;
            return this;
        }

        /**
         * Sets working mode.
         *
         * @param workingMode the working mode
         * @return the working mode
         */
        public Builder setWorkingMode(ICloudBeacon.WorkingMode workingMode) {
            this.workingMode = workingMode;
            return this;
        }

        /**
         * Sets wifi scan interval.
         *
         * @param wifiScanInterval the wifi scan interval
         * @return the wifi scan interval
         */
        public Builder setWifiScanInterval(int wifiScanInterval) {
            this.wifiScanInterval = wifiScanInterval;
            return this;
        }

        /**
         * Sets data send interval.
         *
         * @param dataSendInterval the data send interval
         * @return the data send interval
         */
        public Builder setDataSendInterval(int dataSendInterval) {
            this.dataSendInterval = dataSendInterval;
            return this;
        }

        /**
         * Sets ble scan interval.
         *
         * @param bleScanInterval the ble scan interval
         * @return the ble scan interval
         */
        public Builder setBleScanInterval(int bleScanInterval) {
            this.bleScanInterval = bleScanInterval;
            return this;
        }

        /**
         * Sets ble scan duration.
         *
         * @param bleScanDuration the ble scan duration
         * @return the ble scan duration
         */
        public Builder setBleScanDuration(int bleScanDuration) {
            this.bleScanDuration = bleScanDuration;
            return this;
        }

        /**
         * Sets password.
         *
         * @param password the password
         * @return the password
         */
        public Builder setPassword(String password) {
            this.password = password;
            return this;
        }

        /**
         * Sets name.
         *
         * @param name the name
         * @return the name
         */
        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        /**
         * Build cloud config.
         *
         * @return the cloud config
         */
        public CloudConfig build() {
            return new CloudConfig(this);
        }
    }
}
