package com.kontakt.sdk.android.common.model;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.JSONUtils;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Representation of cloud beacon
 */
public class CloudBeacon extends Device implements ICloudBeacon {

    private final int bleScanDuration;
    private final int bleScanInterval;
    private final int dataSendInterval;
    private final String defaultSSIDAuth;
    private final String defaultSSIDCrypt;
    private final String defaultSSIDName;
    private final HashingPolicy hashingPolicy;
    private final String maintenanceEndTime;
    private final String maintenanceStartTime;
    private final int wifiScanInterval;
    private final WorkingMode workingMode;

    private final int hashCode;

    public static final Parcelable.Creator<CloudBeacon> CREATOR = new Creator<CloudBeacon>() {
        @Override
        public CloudBeacon createFromParcel(Parcel source) {
            Bundle bundle = source.readBundle(getClass().getClassLoader());
            return new CloudBeacon(bundle, source);
        }

        @Override
        public CloudBeacon[] newArray(int size) {
            return new CloudBeacon[size];
        }
    };

    public CloudBeacon(Builder builder) {
        super(builder);
        this.wifiScanInterval = builder.wifiScanInterval;
        this.bleScanDuration = builder.bleScanDuration;
        this.workingMode = builder.workingMode;
        this.defaultSSIDAuth = builder.defaultSSIDAuth;
        this.maintenanceStartTime = builder.maintenanceStartTime;
        this.dataSendInterval = builder.dataSendInterval;
        this.defaultSSIDCrypt = builder.defaultSSIDCrypt;
        this.bleScanInterval = builder.bleScanInterval;
        this.defaultSSIDName = builder.defaultSSIDName;
        this.maintenanceEndTime = builder.maintenanceEndTime;
        this.hashingPolicy = builder.hashingPolicy;

        this.hashCode = calculateHashCode();
    }

    private CloudBeacon(Bundle bundle, Parcel parcel) {
        super(bundle, parcel);
        this.wifiScanInterval = bundle.getInt(Constants.Devices.WIFI_SCAN_INTERVAL);
        this.bleScanDuration = bundle.getInt(Constants.Devices.BLE_SCAN_DURATION);
        this.workingMode = (WorkingMode) bundle.getSerializable(Constants.Devices.WORKING_MODE);
        this.defaultSSIDAuth = bundle.getString(Constants.Devices.DEFAULT_SSID_AUTH);
        this.maintenanceStartTime = bundle.getString(Constants.Devices.MAINTENANCE_START);
        this.maintenanceEndTime = bundle.getString(Constants.Devices.MAINTENANCE_END);
        this.dataSendInterval = bundle.getInt(Constants.Devices.DATA_SEND_INTERVAL);
        this.defaultSSIDCrypt = bundle.getString(Constants.Devices.DEFAULT_SSID_CRYPT);
        this.bleScanInterval = bundle.getInt(Constants.Devices.BLE_SCAN_INTERVAL);
        this.defaultSSIDName = bundle.getString(Constants.Devices.DEFAULT_SSID_NAME);
        this.hashingPolicy = (HashingPolicy) bundle.getSerializable(Constants.Devices.HASHING_POLICY);

        this.hashCode = calculateHashCode();
    }

    @Override
    protected int calculateHashCode() {
        return HashCodeBuilder.init()
                .append(super.hashCode())
                .append(wifiScanInterval)
                .append(bleScanDuration)
                .append(workingMode)
                .append(defaultSSIDAuth)
                .append(maintenanceStartTime)
                .append(dataSendInterval)
                .append(defaultSSIDCrypt)
                .append(bleScanInterval)
                .append(defaultSSIDName)
                .append(maintenanceEndTime)
                .append(hashingPolicy)
                .build();
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (o == null || !(o instanceof CloudBeacon)) {
            return false;
        }

        if (!super.equals(o)) {
            return false;
        }

        CloudBeacon cloudBeacon = (CloudBeacon) o;

        return SDKEqualsBuilder.start()
                .equals(wifiScanInterval, cloudBeacon.wifiScanInterval)
                .equals(bleScanDuration, cloudBeacon.bleScanDuration)
                .equals(workingMode, cloudBeacon.workingMode)
                .equals(defaultSSIDAuth, cloudBeacon.defaultSSIDAuth)
                .equals(maintenanceEndTime, cloudBeacon.maintenanceEndTime)
                .equals(maintenanceStartTime, cloudBeacon.maintenanceStartTime)
                .equals(dataSendInterval, cloudBeacon.dataSendInterval)
                .equals(defaultSSIDCrypt, cloudBeacon.defaultSSIDCrypt)
                .equals(bleScanInterval, cloudBeacon.bleScanInterval)
                .equals(defaultSSIDName, cloudBeacon.defaultSSIDName)
                .equals(hashingPolicy, cloudBeacon.hashingPolicy)
                .result();

    }

    public static class Builder extends Device.Builder<CloudBeacon, Builder> {

        int wifiScanInterval;
        int bleScanDuration;
        WorkingMode workingMode;
        String defaultSSIDAuth;
        String maintenanceStartTime;
        int dataSendInterval;
        String defaultSSIDCrypt;
        int bleScanInterval;
        String defaultSSIDName;
        String maintenanceEndTime;
        HashingPolicy hashingPolicy;

        /**
         * Sets wifi scan interval.
         *
         * @param wifiScanInterval the wifi scan interval
         * @return Builder instance
         */
        public Builder setWifiScanInterval(int wifiScanInterval) {
            this.wifiScanInterval = wifiScanInterval;
            return this;
        }

        /**
         * Sets ble scan duration.
         *
         * @param bleScanDuration the ble scan duration
         * @return Builder instance
         */
        public Builder setBleScanDuration(int bleScanDuration) {
            this.bleScanDuration = bleScanDuration;
            return this;
        }

        /**
         * Sets working mode.
         *
         * @param workingMode the working mode
         * @return Builder instance
         */
        public Builder setWorkingMode(WorkingMode workingMode) {
            this.workingMode = workingMode;
            return this;
        }

        /**
         * Sets default sSID auth.
         *
         * @param defaultSSIDAuth the default SSID auth
         * @return Builder instance
         */
        public Builder setDefaultSSIDAuth(String defaultSSIDAuth) {
            this.defaultSSIDAuth = defaultSSIDAuth;
            return this;
        }

        /**
         * Sets maintenance start time.
         *
         * @param maintenanceStartTime the maintenance start time
         * @return Builder instance
         */
        public Builder setMaintenanceStart(final String maintenanceStartTime) {
            this.maintenanceStartTime = maintenanceStartTime;
            return this;
        }

        /**
         * Sets hashing policy.
         *
         * @param hashingPolicy the hashing policy
         * @return the hashing policy
         */
        public Builder setHashingPolicy(HashingPolicy hashingPolicy) {
            this.hashingPolicy = hashingPolicy;
            return this;
        }

        /**
         * Sets data send interval.
         *
         * @param dataSendInterval the data send interval
         * @return Builder instance
         */
        public Builder setDataSendInterval(int dataSendInterval) {
            this.dataSendInterval = dataSendInterval;
            return this;
        }

        /**
         * Sets default sSID crypt.
         *
         * @param defaultSSIDCrypt the default sSID crypt
         * @return Builder instance
         */
        public Builder setDefaultSSIDCrypt(String defaultSSIDCrypt) {
            this.defaultSSIDCrypt = defaultSSIDCrypt;
            return this;
        }

        /**
         * Sets ble scan interval.
         *
         * @param bleScanInterval the ble scan interval
         * @return Builder instance
         */
        public Builder setBleScanInterval(int bleScanInterval) {
            this.bleScanInterval = bleScanInterval;
            return this;
        }

        /**
         * Sets default sSID name.
         *
         * @param defaultSSIDName the default sSID name
         * @return Builder instance
         */
        public Builder setDefaultSSIDName(String defaultSSIDName) {
            this.defaultSSIDName = defaultSSIDName;
            return this;
        }

        /**
         * Sets maintenance end time.
         *
         * @param maintenanceEndTime the maintenance end time
         * @return Builder instance
         */
        public Builder setMaintenanceEnd(String maintenanceEndTime) {
            this.maintenanceEndTime = maintenanceEndTime;
            return this;
        }


        @Override
        public CloudBeacon build() {
            return new CloudBeacon(this);
        }
    }

    public static CloudBeacon from(JSONObject jsonObject) {
        final Access access = Access.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.ACCESS));
        final String hexedUrl = JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.URL);
        final DeviceType deviceType = DeviceType.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEVICE_TYPE));
        final HashingPolicy hashingPolicy = HashingPolicy.valueOf(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.HASHING_POLICY));
        try {
            JSONArray profiles = jsonObject.optJSONArray(Constants.Devices.PROFILES);

            CloudBeacon.Builder builder = new CloudBeacon.Builder();

            builder.setWifiScanInterval(JSONUtils.getInt(jsonObject, Constants.Devices.WIFI_SCAN_INTERVAL, 0))
                    .setBleScanDuration(JSONUtils.getInt(jsonObject, Constants.Devices.BLE_SCAN_DURATION, 0))
                    .setWorkingMode(WorkingMode.valueOf(JSONUtils.getString(jsonObject, Constants.Devices.WORKING_MODE, WorkingMode.NONE.name())))
                    .setDefaultSSIDAuth(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_AUTH))
                    .setMaintenanceStart(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.MAINTENANCE_START))
                    .setMaintenanceEnd(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.MAINTENANCE_END))
                    .setDataSendInterval(JSONUtils.getInt(jsonObject, Constants.Devices.DATA_SEND_INTERVAL, 0))
                    .setDefaultSSIDName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_NAME))
                    .setDefaultSSIDCrypt(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.DEFAULT_SSID_CRYPT))
                    .setBleScanInterval(JSONUtils.getInt(jsonObject, Constants.Devices.BLE_SCAN_INTERVAL, 0))
                    .setHashingPolicy(hashingPolicy)
                    .setId(JSONUtils.getUUIDOrNull(jsonObject, Constants.ID))
                    .setProximityUUID(JSONUtils.getUUIDOrNull(jsonObject, Constants.Devices.PROXIMITY))
                    .setMajor(JSONUtils.getInt(jsonObject, Constants.Devices.MAJOR, 0))
                    .setMinor(JSONUtils.getInt(jsonObject, Constants.Devices.MINOR, 0))
                    .setMacAddress(JSONUtils.getString(jsonObject, Constants.Devices.MAC_ADDRESS, ""))
                    .setTxPower(JSONUtils.getInt(jsonObject, Constants.Devices.TX_POWER, -1))
                    .setName(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.NAME))
                    .setAlias(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.ALIAS))
                    .setInterval(JSONUtils.getInt(jsonObject, Constants.Devices.INTERVAL, 0))
                    .setUniqueId(JSONUtils.getString(jsonObject, Constants.UNIQUE_ID, ""))
                    .setActionsCount(JSONUtils.getInt(jsonObject, Constants.Devices.ACTIONS_COUNT, 0))
                    .setManagerId(JSONUtils.getUUIDOrNull(jsonObject, Constants.Devices.MANAGER_ID))
                    .setVenue(JSONUtils.hasJSONKey(jsonObject, Constants.VENUE) ? Venue.from(jsonObject.getJSONObject(Constants.VENUE)) : null)
                    .setLatitude(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.LATITUDE))
                    .setLongitude(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.LONGITUDE))
                    .setAccess(access)
                    .setFirmwareVersion(JSONUtils.getStringOrNull(jsonObject, Constants.Devices.FIRMWARE_VERSION))
                    .setInstanceId(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.INSTANCE_ID))
                    .setUrl(hexedUrl != null ? EddystoneUtils.fromHexedUrlToUrl(hexedUrl) : null)
                    .setNamespace(JSONUtils.getStringOrNull(jsonObject, Constants.Eddystone.NAMESPACE))
                    .setDeviceType(deviceType)
            ;


            if (profiles != null) {
                List<DeviceProfile> deviceProfileList = new ArrayList<DeviceProfile>();
                int profilesLength = profiles.length();
                for (int i = 0; i < profilesLength; i++) {
                    deviceProfileList.add(DeviceProfile.valueOf((String) profiles.get(i)));
                }
                builder.addDeviceProfiles(deviceProfileList);
            }

            return builder.build();

        } catch (JSONException e) {
            throw new IllegalStateException(e);
        }

    }

    @Override
    public int getWifiScanInterval() {
        return wifiScanInterval;
    }

    @Override
    public int getBLEScanDuration() {
        return bleScanDuration;
    }

    @Override
    public WorkingMode getWorkingMode() {
        return workingMode;
    }

    @Override
    public String getDefaultSSIDAuth() {
        return defaultSSIDAuth;
    }

    @Override
    public String getMaintenanceStart() {
        return maintenanceStartTime;
    }

    @Override
    public int getDataSendInterval() {
        return dataSendInterval;
    }


    @Override
    public String getDefaultSSIDCrypt() {
        return defaultSSIDCrypt;
    }

    @Override
    public int getBleScanInterval() {
        return bleScanInterval;
    }

    @Override
    public String getDefaultSSIDName() {
        return defaultSSIDName;
    }

    @Override
    public String getMaintenanceEnd() {
        return maintenanceEndTime;
    }

    @Override
    public HashingPolicy getHashingPolicy() {
        return hashingPolicy;
    }

    @Override
    public boolean isShuffled() {
        return false;
    }

    @Override
    public UUID getSecureProximityUUID() {
        return null;
    }

    @Override
    public String getSecureNamespace() {
        return null;
    }

    @Override
    protected void parcelProperties(Bundle bundle) {
        bundle.putInt(Constants.Devices.WIFI_SCAN_INTERVAL, getWifiScanInterval());
        bundle.putInt(Constants.Devices.BLE_SCAN_DURATION, getBLEScanDuration());
        bundle.putSerializable(Constants.Devices.WORKING_MODE, getWorkingMode());
        bundle.putSerializable(Constants.Devices.HASHING_POLICY, getHashingPolicy());
        bundle.putString(Constants.Devices.DEFAULT_SSID_AUTH, getDefaultSSIDAuth());
        bundle.putString(Constants.Devices.MAINTENANCE_START, getMaintenanceStart());
        bundle.putInt(Constants.Devices.DATA_SEND_INTERVAL, getDataSendInterval());
        bundle.putString(Constants.Devices.DEFAULT_SSID_CRYPT, getDefaultSSIDCrypt());
        bundle.putInt(Constants.Devices.BLE_SCAN_INTERVAL, getBleScanInterval());
        bundle.putString(Constants.Devices.DEFAULT_SSID_NAME, getDefaultSSIDName());
        bundle.putString(Constants.Devices.MAINTENANCE_END, getMaintenanceEnd());
    }
}
